import React from 'react';
import { Transaction } from '../../types';
import { ArrowDownLeft, RefreshCw, CreditCard, DollarSign, Activity, Smartphone, Building2, Zap } from 'lucide-react';

interface RecentTransactionsProps {
  transactions: Transaction[];
}

const RecentTransactions: React.FC<RecentTransactionsProps> = ({ transactions }) => {
  const getTransactionIcon = (type: string) => {
    switch (type) {
      case 'credit':
      case 'recharge_cinetpay':
        return <ArrowDownLeft className="w-4 h-4 text-amber-600" />;
      case 'debit':
        return <DollarSign className="w-4 h-4 text-red-600" />;
      case 'nfc_deposit':
      case 'nfc_withdrawal':
        return <Zap className="w-4 h-4 text-yellow-600" />;
      case 'transfer':
      case 'mobile_transfer':
        return <Smartphone className="w-4 h-4 text-orange-600" />;
      case 'loan':
      case 'pret_mobile':
        return <CreditCard className="w-4 h-4 text-amber-700" />;
      case 'repayment':
      case 'remboursement_auto':
        return <RefreshCw className="w-4 h-4 text-orange-600" />;
      case 'recharge_by_business':
        return <Activity className="w-4 h-4 text-amber-500" />;
      case 'virement_bancaire':
        return <Building2 className="w-4 h-4 text-amber-800" />;
      default:
        return <DollarSign className="w-4 h-4 text-amber-600" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-amber-100 text-amber-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      case 'cancelled':
        return 'bg-orange-100 text-orange-800';
      default:
        return 'bg-amber-100 text-amber-800';
    }
  };

  const formatAmount = (amount: number, type: string) => {
    const prefix = type === 'credit' || type === 'loan' || type === 'recharge_cinetpay' || type === 'pret_mobile' ? '+' : '-';
    return `${prefix}${amount.toLocaleString()} FCFA`;
  };

  const getTransactionLabel = (type: string) => {
    switch (type) {
      case 'recharge_cinetpay':
        return 'Recharge CinetPay';
      case 'nfc_deposit':
        return 'Dépôt NFC';
      case 'nfc_withdrawal':
        return 'Retrait NFC';
      case 'mobile_transfer':
        return 'Transfert Mobile';
      case 'pret_mobile':
        return 'Prêt Mobile';
      case 'remboursement_auto':
        return 'Remboursement Auto';
      case 'recharge_by_business':
        return 'Recharge par TIM BUSINESS';
      case 'virement_bancaire':
        return 'Virement Bancaire';
      case 'credit':
        return 'Crédit';
      case 'debit':
        return 'Débit';
      case 'transfer':
        return 'Transfert';
      case 'loan':
        return 'Prêt';
      case 'repayment':
        return 'Remboursement';
      default:
        return 'Transaction';
    }
  };

  const getTimTypeLabel = (timType?: string) => {
    switch (timType) {
      case 'TIM_MINI':
        return 'TIM MINI';
      case 'TIM_MAXI':
        return 'TIM MAXI';
      case 'TIM_BUSINESS':
        return 'TIM BUSINESS';
      default:
        return '';
    }
  };

  return (
    <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 rounded-xl shadow-sm border border-amber-200/30">
      <div className="p-4 border-b border-amber-200/50">
        <h3 className="text-base font-semibold text-amber-900">Transactions Mobiles Récentes</h3>
        <p className="text-xs text-amber-600">Activités TIM CASH - NFC, CinetPay, Prêts</p>
      </div>
      <div className="divide-y divide-amber-200/50">
        {transactions.map((transaction) => (
          <div key={transaction.id} className="p-4 hover:bg-amber-50/50 transition-colors duration-150">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-3">
                <div className="w-8 h-8 bg-amber-100 rounded-lg flex items-center justify-center">
                  {getTransactionIcon(transaction.type)}
                </div>
                <div>
                  <div className="flex items-center space-x-2">
                    <p className="font-medium text-amber-900 text-sm">
                      {getTransactionLabel(transaction.type)}
                    </p>
                    {/* Badge pour méthode de rechargement */}
                    {transaction.rechargeMethod && (
                      <span className={`inline-flex px-1.5 py-0.5 text-xs font-medium rounded ${
                        transaction.rechargeMethod === 'cinetpay' 
                          ? 'bg-amber-100 text-amber-800' 
                          : 'bg-orange-100 text-orange-800'
                      }`}>
                        {transaction.rechargeMethod === 'cinetpay' ? 'CinetPay' : 'TIM BUSINESS'}
                      </span>
                    )}
                    {/* Badge pour types de comptes */}
                    {(transaction.fromWalletType || transaction.toWalletType) && (
                      <span className="inline-flex px-1.5 py-0.5 text-xs font-medium bg-orange-100 text-orange-800 rounded">
                        {transaction.fromWalletType ? getTimTypeLabel(transaction.fromWalletType) : ''}
                        {transaction.fromWalletType && transaction.toWalletType ? ' → ' : ''}
                        {transaction.toWalletType ? getTimTypeLabel(transaction.toWalletType) : ''}
                      </span>
                    )}
                  </div>
                  <p className="text-xs text-amber-600">{transaction.description}</p>
                  <p className="text-xs text-amber-500">
                    {new Date(transaction.createdAt).toLocaleDateString('fr-FR')} à{' '}
                    {new Date(transaction.createdAt).toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' })}
                  </p>
                </div>
              </div>
              <div className="text-right">
                <p className={`font-semibold text-sm ${
                  ['credit', 'loan', 'recharge_cinetpay', 'pret_mobile', 'nfc_deposit', 'recharge_by_business'].includes(transaction.type)
                    ? 'text-amber-600' 
                    : 'text-red-600'
                }`}>
                  {formatAmount(transaction.amount, transaction.type)}
                </p>
                <span className={`inline-flex px-2 py-0.5 text-xs font-medium rounded-full ${getStatusColor(transaction.status)}`}>
                  {transaction.status === 'completed' ? 'Terminé' : 
                   transaction.status === 'pending' ? 'En cours' :
                   transaction.status === 'failed' ? 'Échoué' : transaction.status}
                </span>
              </div>
            </div>
          </div>
        ))}
      </div>
      {transactions.length === 0 && (
        <div className="p-6 text-center">
          <Activity className="w-10 h-10 text-amber-400 mx-auto mb-3" />
          <p className="text-amber-600 text-sm">Aucune transaction mobile récente</p>
          <p className="text-amber-500 text-xs">Les paiements NFC et recharges CinetPay apparaîtront ici</p>
        </div>
      )}
    </div>
  );
};

export default RecentTransactions;