import React from 'react';
import { LucideIcon } from 'lucide-react';
import { Link } from 'react-router-dom';

interface QuickAction {
  name: string;
  icon: LucideIcon;
  color: 'blue' | 'green' | 'purple' | 'orange' | 'red' | 'amber' | 'gold' | 'copper';
  href: string;
  description: string;
  onClick?: () => void;
}

interface QuickActionsProps {
  actions: QuickAction[];
  title?: string;
}

const QuickActions: React.FC<QuickActionsProps> = ({ 
  actions, 
  title = "Actions Rapides" 
}) => {
  const colorClasses = {
    blue: 'hover:bg-amber-50 hover:text-amber-700 bg-amber-100 text-amber-600',
    green: 'hover:bg-yellow-50 hover:text-yellow-700 bg-yellow-100 text-yellow-600',
    purple: 'hover:bg-orange-50 hover:text-orange-700 bg-orange-100 text-orange-600',
    orange: 'hover:bg-amber-50 hover:text-amber-700 bg-amber-100 text-amber-600',
    red: 'hover:bg-red-50 hover:text-red-700 bg-red-100 text-red-600',
    amber: 'hover:bg-amber-50 hover:text-amber-700 bg-amber-100 text-amber-600',
    gold: 'hover:bg-yellow-50 hover:text-yellow-700 bg-yellow-100 text-yellow-600',
    copper: 'hover:bg-orange-50 hover:text-orange-700 bg-orange-100 text-orange-600'
  };

  return (
    <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-4">
      <h3 className="text-base font-bold text-amber-900 mb-4">{title}</h3>
      <div className="space-y-2">
        {actions.map((action, index) => {
          const Icon = action.icon;
          const colorClass = colorClasses[action.color];
          
          const ActionContent = (
            <>
              <div className={`p-2 rounded-lg ${colorClass} group-hover:scale-110 transition-transform shadow-sm`}>
                <Icon className="w-4 h-4" />
              </div>
              <div className="flex-1">
                <div className="font-medium text-amber-900 group-hover:text-amber-700 text-sm">
                  {action.name}
                </div>
                <div className="text-xs text-amber-600 group-hover:text-amber-700">
                  {action.description}
                </div>
              </div>
              <div className="opacity-0 group-hover:opacity-100 transition-opacity">
                <svg className="w-4 h-4 text-amber-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                </svg>
              </div>
            </>
          );
          
          if (action.onClick) {
            return (
              <button
                key={index}
                onClick={action.onClick}
                className="group w-full text-left p-3 hover:bg-amber-50 rounded-lg border border-amber-100 hover:border-amber-200 transition-all duration-200 flex items-center space-x-3 hover:shadow-md"
              >
                {ActionContent}
              </button>
            );
          }
          
          return (
            <Link
              key={index}
              to={action.href}
              className="group w-full text-left p-3 hover:bg-amber-50 rounded-lg border border-amber-100 hover:border-amber-200 transition-all duration-200 flex items-center space-x-3 hover:shadow-md"
            >
              {ActionContent}
            </Link>
          );
        })}
      </div>
    </div>
  );
};

export default QuickActions;
