import React, { useState, useEffect } from 'react';
import { BarChart3, LineChart, Activity, CreditCard, Wallet, TrendingUp, TrendingDown, Zap, DollarSign } from 'lucide-react';

interface ChartData {
  label: string;
  value: number;
  change?: number;
}

interface PerformanceMetrics {
  nfc_transactions?: number;
  mobile_loans?: number;
  cinetpay_recharges?: number;
  total_volume?: number;
  total_transactions?: number;
}

interface PerformanceChartProps {
  title?: string;
  data: ChartData[];
  type?: 'bar' | 'line';
  color?: 'blue' | 'green' | 'purple' | 'orange';
  showControls?: boolean;
  metrics?: PerformanceMetrics;
}

const PerformanceChart: React.FC<PerformanceChartProps> = ({
  title = "Performance",
  data,
  type = 'bar',
  color = 'blue',
  showControls = true,
  metrics
}) => {
  const [chartType, setChartType] = useState<'bar' | 'line'>(type);
  const [selectedMetric, setSelectedMetric] = useState('paiements-nfc');
  const [animatedValues, setAnimatedValues] = useState<number[]>([]);
  const [hoveredIndex, setHoveredIndex] = useState<number | null>(null);

  // Animation des valeurs au chargement
  useEffect(() => {
    const timer = setTimeout(() => {
      setAnimatedValues(data.map(d => d.value));
    }, 100);
    return () => clearTimeout(timer);
  }, [data]);

  const colorClasses = {
    blue: {
      gradient: 'from-amber-500 via-orange-600 to-amber-700',
      lightGradient: 'from-amber-100 via-orange-200 to-amber-300',
      text: 'text-amber-600',
      bg: 'bg-amber-100',
      shadow: 'shadow-amber-200/50',
      glow: 'shadow-amber-500/30'
    },
    green: {
      gradient: 'from-yellow-400 via-amber-500 to-orange-600',
      lightGradient: 'from-yellow-100 via-amber-200 to-orange-300',
      text: 'text-amber-700',
      bg: 'bg-amber-100',
      shadow: 'shadow-amber-200/50',
      glow: 'shadow-amber-500/30'
    },
    purple: {
      gradient: 'from-orange-500 via-red-500 to-orange-600',
      lightGradient: 'from-orange-100 via-red-200 to-orange-300',
      text: 'text-orange-600',
      bg: 'bg-orange-100',
      shadow: 'shadow-orange-200/50',
      glow: 'shadow-orange-500/30'
    },
    orange: {
      gradient: 'from-amber-600 via-orange-600 to-red-600',
      lightGradient: 'from-amber-100 via-orange-200 to-red-300',
      text: 'text-orange-600',
      bg: 'bg-orange-100',
      shadow: 'shadow-orange-200/50',
      glow: 'shadow-orange-500/30'
    }
  };

  const colors = colorClasses[color];
  const maxValue = Math.max(...data.map(d => d.value));

  const metricsData = [
    { 
      id: 'paiements-nfc', 
      label: 'Paiements NFC', 
      icon: Zap,
      value: metrics?.nfc_transactions?.toString() || '0',
      change: '0%',
      trend: 'up'
    },
    { 
      id: 'prets-mobiles', 
      label: 'Prêts Mobiles', 
      icon: CreditCard,
      value: metrics?.mobile_loans?.toString() || '0',
      change: '0%',
      trend: 'up'
    },
    { 
      id: 'recharges-cinetpay', 
      label: 'Recharges CinetPay', 
      icon: DollarSign,
      value: metrics?.cinetpay_recharges?.toString() || '0',
      change: '0%',
      trend: 'up'
    }
  ];

  return (
    <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-4">
      {/* Header */}
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-base font-bold text-amber-900">{title}</h3>
        
        {showControls && (
          <div className="flex items-center space-x-3">
            {/* Chart Type Toggle avec design moderne */}
            <div className="flex items-center space-x-1 bg-gradient-to-r from-amber-100 to-orange-200 rounded-xl p-1 shadow-inner">
              <button
                onClick={() => setChartType('bar')}
                className={`p-2 rounded-lg transition-all duration-300 transform ${
                  chartType === 'bar'
                    ? `bg-white shadow-lg ${colors.glow} scale-105 ${colors.text}`
                    : 'text-amber-500 hover:text-amber-700 hover:bg-white/50'
                }`}
              >
                <BarChart3 className="w-4 h-4" />
              </button>
              <button
                onClick={() => setChartType('line')}
                className={`p-2 rounded-lg transition-all duration-300 transform ${
                  chartType === 'line'
                    ? `bg-white shadow-lg ${colors.glow} scale-105 ${colors.text}`
                    : 'text-amber-500 hover:text-amber-700 hover:bg-white/50'
                }`}
              >
                <LineChart className="w-4 h-4" />
              </button>
            </div>
          </div>
        )}
      </div>

      {/* Métriques en haut */}
      <div className="grid grid-cols-3 gap-3 mb-6">
        {metricsData.map((metric, index) => {
          const Icon = metric.icon;
          const TrendIcon = metric.trend === 'up' ? TrendingUp : TrendingDown;
          return (
            <div 
              key={metric.id}
              className={`p-3 rounded-xl bg-gradient-to-br ${colors.lightGradient} border border-white/50 shadow-sm hover:shadow-md transition-all duration-300 cursor-pointer transform hover:scale-105`}
              onClick={() => setSelectedMetric(metric.id)}
            >
              <div className="flex items-center justify-between mb-2">
                <Icon className={`w-5 h-5 ${colors.text}`} />
                <TrendIcon className="w-4 h-4 text-green-600" />
              </div>
              <p className="text-lg font-bold text-gray-900">{metric.value}</p>
              <p className="text-xs text-gray-600 mb-1">{metric.label}</p>
              <p className="text-xs font-semibold text-green-600">{metric.change}</p>
            </div>
          );
        })}
      </div>

      {/* Chart */}
      <div className="h-56 relative bg-gradient-to-br from-gray-50 to-white rounded-xl p-4 shadow-inner">
        {chartType === 'bar' ? (
          // Bar Chart Moderne avec animations
          <div className="h-full flex items-end justify-between space-x-3">
            {data.map((item, index) => {
              const animatedHeight = animatedValues[index] || 0;
              const height = (animatedHeight / maxValue) * 100;
              const isHovered = hoveredIndex === index;
              
              return (
                <div 
                  key={index} 
                  className="flex-1 flex flex-col items-center group"
                  onMouseEnter={() => setHoveredIndex(index)}
                  onMouseLeave={() => setHoveredIndex(null)}
                >
                  <div className="relative w-full">
                    {/* Tooltip Moderne */}
                    <div className={`absolute -top-16 left-1/2 transform -translate-x-1/2 bg-gray-900 text-white text-xs px-3 py-2 rounded-xl shadow-xl transition-all duration-300 whitespace-nowrap z-20 ${
                      isHovered ? 'opacity-100 scale-100' : 'opacity-0 scale-95'
                    }`}>
                      <div className="text-center">
                        <p className="font-semibold">{item.label}</p>
                        <p className="text-blue-300">{(item.value / 1000000).toFixed(1)}M FCFA</p>
                        {item.change && (
                          <p className={`text-xs ${item.change > 0 ? 'text-green-300' : 'text-red-300'}`}>
                            {item.change > 0 ? '+' : ''}{item.change}%
                          </p>
                        )}
                      </div>
                      {/* Flèche du tooltip */}
                      <div className="absolute top-full left-1/2 transform -translate-x-1/2 w-0 h-0 border-l-4 border-r-4 border-t-4 border-transparent border-t-gray-900"></div>
                    </div>
                    
                    {/* Barre avec effet de lueur */}
                    <div className="relative">
                      {/* Effet de lueur en arrière-plan */}
                      <div 
                        className={`absolute inset-0 bg-gradient-to-t ${colors.gradient} rounded-t-xl opacity-20 blur-sm transition-all duration-700`}
                        style={{ height: `${height + 10}%` }}
                      />
                      
                      {/* Barre principale */}
                      <div 
                        className={`relative w-full bg-gradient-to-t ${colors.gradient} rounded-t-xl transition-all duration-700 cursor-pointer shadow-lg hover:shadow-xl transform ${
                          isHovered ? 'scale-105' : 'scale-100'
                        }`}
                        style={{ 
                          height: `${height}%`,
                          boxShadow: isHovered ? `0 10px 25px ${colors.shadow}` : undefined
                        }}
                      >
                        {/* Reflet sur la barre */}
                        <div className="absolute inset-0 bg-gradient-to-t from-transparent via-white/20 to-white/40 rounded-t-xl"></div>
                      </div>
                    </div>
                  </div>
                  
                  {/* Label avec style moderne */}
                  <span className={`text-xs mt-3 text-center font-medium transition-colors duration-200 ${
                    isHovered ? colors.text : 'text-gray-500'
                  }`}>
                    {item.label}
                  </span>
                </div>
              );
            })}
          </div>
        ) : (
          // Line Chart Moderne avec animations
          <div className="h-full relative">
            <svg className="w-full h-full" viewBox="0 0 400 200">
              <defs>
                <linearGradient id={`gradient-${color}`} x1="0%" y1="0%" x2="0%" y2="100%">
                  <stop offset="0%" stopColor="#3b82f6" stopOpacity="0.8" />
                  <stop offset="100%" stopColor="#1d4ed8" stopOpacity="0.1" />
                </linearGradient>
                <filter id="glow">
                  <feGaussianBlur stdDeviation="3" result="coloredBlur"/>
                  <feMerge> 
                    <feMergeNode in="coloredBlur"/>
                    <feMergeNode in="SourceGraphic"/>
                  </feMerge>
                </filter>
              </defs>
              
              {/* Grid Lines avec style moderne */}
              {[0, 1, 2, 3, 4].map(i => (
                <line
                  key={i}
                  x1="0"
                  y1={i * 40}
                  x2="400"
                  y2={i * 40}
                  stroke="#e5e7eb"
                  strokeWidth="1"
                  strokeDasharray="2,2"
                  opacity="0.5"
                />
              ))}
              
              {/* Zone sous la courbe */}
              <path
                d={`M 0,200 L ${data.map((item, index) => 
                  `${(index * 400) / (data.length - 1)},${200 - (item.value / maxValue) * 180}`
                ).join(' L ')} L 400,200 Z`}
                fill={`url(#gradient-${color})`}
                opacity="0.3"
              />
              
              {/* Line Path avec effet de lueur */}
              <path
                d={`M ${data.map((item, index) => 
                  `${(index * 400) / (data.length - 1)},${200 - (item.value / maxValue) * 180}`
                ).join(' L ')}`}
                fill="none"
                stroke="#3b82f6"
                strokeWidth="4"
                filter="url(#glow)"
                className="drop-shadow-lg"
              />
              
              {/* Data Points avec animations */}
              {data.map((item, index) => (
                <g key={index}>
                  {/* Cercle de lueur */}
                  <circle
                    cx={(index * 400) / (data.length - 1)}
                    cy={200 - (item.value / maxValue) * 180}
                    r="8"
                    fill="#3b82f6"
                    opacity="0.2"
                    className="animate-pulse"
                  />
                  {/* Point principal */}
                  <circle
                    cx={(index * 400) / (data.length - 1)}
                    cy={200 - (item.value / maxValue) * 180}
                    r="5"
                    fill="#ffffff"
                    stroke="#3b82f6"
                    strokeWidth="3"
                    className="hover:r-7 transition-all cursor-pointer shadow-lg"
                  />
                </g>
              ))}
            </svg>
          </div>
        )}
      </div>

      {/* Summary Stats Modernisées */}
      <div className="mt-6 grid grid-cols-3 gap-4">
        <div className="text-center p-4 bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl border border-blue-200/50">
          <div className="flex items-center justify-center mb-2">
            <DollarSign className="w-5 h-5 text-blue-600 mr-1" />
          </div>
          <p className="text-xl font-bold text-blue-900">
            {(data.reduce((sum, item) => sum + item.value, 0) / 1000000000).toFixed(1)}Mrd
          </p>
          <p className="text-xs text-blue-600 font-medium">Volume Total FCFA</p>
        </div>
        
        <div className="text-center p-4 bg-gradient-to-br from-green-50 to-green-100 rounded-xl border border-green-200/50">
          <div className="flex items-center justify-center mb-2">
            <TrendingUp className="w-5 h-5 text-green-600 mr-1" />
          </div>
          <p className="text-xl font-bold text-green-900">
            {(data.reduce((sum, item) => sum + item.value, 0) / data.length / 1000000).toFixed(1)}M
          </p>
          <p className="text-xs text-green-600 font-medium">Moyenne Mensuelle</p>
        </div>
        
        <div className="text-center p-4 bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl border border-purple-200/50">
          <div className="flex items-center justify-center mb-2">
            <Zap className="w-5 h-5 text-purple-600 mr-1" />
          </div>
          <p className="text-xl font-bold text-purple-900">
            +15.3%
          </p>
          <p className="text-xs text-purple-600 font-medium">Croissance Mobile</p>
        </div>
      </div>
    </div>
  );
};

export default PerformanceChart;
