import React from 'react';
import { AlertTriangle, Bell, CheckCircle, Clock, X } from 'lucide-react';

interface Alert {
  id: number;
  type: 'warning' | 'info' | 'success' | 'error';
  title: string;
  message: string;
  time: string;
  action?: string;
  onAction?: () => void;
  onDismiss?: () => void;
}

interface AlertsWidgetProps {
  alerts: Alert[];
  title?: string;
  maxVisible?: number;
}

const AlertsWidget: React.FC<AlertsWidgetProps> = ({ 
  alerts, 
  title = "Alertes", 
  maxVisible = 5 
}) => {
  const getAlertIcon = (type: Alert['type']) => {
    switch (type) {
      case 'warning':
        return <AlertTriangle className="w-5 h-5 text-amber-600" />;
      case 'info':
        return <Bell className="w-5 h-5 text-orange-600" />;
      case 'success':
        return <CheckCircle className="w-5 h-5 text-yellow-600" />;
      case 'error':
        return <X className="w-5 h-5 text-red-600" />;
      default:
        return <Bell className="w-5 h-5 text-amber-600" />;
    }
  };

  const getAlertStyles = (type: Alert['type']) => {
    switch (type) {
      case 'warning':
        return 'bg-amber-50 border-amber-200 hover:bg-amber-100';
      case 'info':
        return 'bg-orange-50 border-orange-200 hover:bg-orange-100';
      case 'success':
        return 'bg-yellow-50 border-yellow-200 hover:bg-yellow-100';
      case 'error':
        return 'bg-red-50 border-red-200 hover:bg-red-100';
      default:
        return 'bg-gray-50 border-gray-200 hover:bg-gray-100';
    }
  };

  const visibleAlerts = alerts.slice(0, maxVisible);

  return (
    <div className="bg-gradient-to-br from-amber-50/90 to-orange-50/90 backdrop-blur-sm rounded-xl shadow-lg border border-amber-200/30 p-4">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-base font-bold text-amber-900">{title}</h3>
        {alerts.length > 0 && (
          <span className="bg-amber-100 text-amber-700 text-xs px-3 py-1 rounded-full font-semibold">
            {alerts.length} nouvelle{alerts.length > 1 ? 's' : ''}
          </span>
        )}
      </div>

      {visibleAlerts.length === 0 ? (
        <div className="text-center py-6">
          <Bell className="w-8 h-8 text-amber-300 mx-auto mb-2" />
          <p className="text-amber-600 text-sm">Aucune alerte pour le moment</p>
        </div>
      ) : (
        <div className="space-y-2">
          {visibleAlerts.map((alert) => (
            <div
              key={alert.id}
              className={`p-3 rounded-lg border transition-all duration-200 ${getAlertStyles(alert.type)}`}
            >
              <div className="flex items-start space-x-2">
                <div className="flex-shrink-0 mt-0.5">
                  {getAlertIcon(alert.type)}
                </div>
                <div className="flex-1 min-w-0">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <h4 className="text-xs font-semibold text-gray-900 mb-1">
                        {alert.title}
                      </h4>
                      <p className="text-xs text-gray-600 mb-2">
                        {alert.message}
                      </p>
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-1 text-xs text-gray-500">
                          <Clock className="w-3 h-3" />
                          <span>Il y a {alert.time}</span>
                        </div>
                        {alert.action && (
                          <button
                            onClick={alert.onAction}
                            className="text-xs font-medium text-blue-600 hover:text-blue-700 transition-colors"
                          >
                            {alert.action}
                          </button>
                        )}
                      </div>
                    </div>
                    {alert.onDismiss && (
                      <button
                        onClick={alert.onDismiss}
                        className="ml-2 p-1 hover:bg-gray-200 rounded-full transition-colors"
                      >
                        <X className="w-3 h-3 text-gray-400" />
                      </button>
                    )}
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      )}

      {alerts.length > maxVisible && (
        <div className="mt-3 text-center">
          <button className="text-xs text-blue-600 hover:text-blue-700 font-medium">
            Voir toutes les alertes ({alerts.length})
          </button>
        </div>
      )}
    </div>
  );
};

export default AlertsWidget;
