import React from 'react';
import { BrowserRouter as Router, Routes, Route, Navigate } from 'react-router-dom';
import { AuthProvider, useAuth } from './contexts/AuthContext';
import Layout from './components/Layout/Layout';
import Login from './pages/Auth/Login';
import Dashboard from './pages/Dashboard/Dashboard';
import Wallet from './pages/Wallet/Wallet';
import Loans from './pages/Loans/Loans';
import Transactions from './pages/Transactions/Transactions';
import Users from './pages/Users/Users';
import AdminStock from './pages/AdminStock/AdminStock';
import Reports from './pages/Reports/Reports';
import Analytics from './pages/Analytics/Analytics';
import Notifications from './pages/Notifications/Notifications';
import Settings from './pages/Settings/Settings';
import CommissionManagement from './pages/Admin/CommissionManagement';
import CardManagement from './pages/Admin/CardManagement';
import VirtualCardPurchases from './pages/Admin/VirtualCardPurchases';
import Workflows from './pages/Workflows/Workflows';
// Pages des sous-menus
import Wallets from './pages/Wallet/Wallets';
import Recharges from './pages/Wallet/Recharges';
import Transfers from './pages/Wallet/Transfers';
import ActiveLoans from './pages/Loans/ActiveLoans';
import PendingLoans from './pages/Loans/PendingLoans';
import LoanRepayments from './pages/Loans/LoanRepayments';
import NFCAnalytics from './pages/Analytics/NFCAnalytics';
import CinetPayReports from './pages/Reports/CinetPayReports';
// Nouvelles pages
import Countries from './pages/Countries/Countries';
import Advertising from './pages/Advertising/Advertising';
import StockAccount from './pages/StockAccount/StockAccount';
import LoansManagement from './pages/LoansManagement/LoansManagement';
import TIMBusinessRecharges from './pages/TIMBusiness/TIMBusinessRecharges';
import Monitoring from './pages/Monitoring/Monitoring';

const ProtectedRoute: React.FC<{ children: React.ReactNode }> = ({ children }) => {
  const { isAuthenticated } = useAuth();
  return isAuthenticated ? <>{children}</> : <Navigate to="/login" />;
};

const PublicRoute: React.FC<{ children: React.ReactNode }> = ({ children }) => {
  const { isAuthenticated } = useAuth();
  return !isAuthenticated ? <>{children}</> : <Navigate to="/dashboard" />;
};

function AppContent() {
  return (
    <Router>
      <Routes>
        <Route path="/login" element={
          <PublicRoute>
            <Login />
          </PublicRoute>
        } />
        
        <Route path="/" element={
          <ProtectedRoute>
            <Layout />
          </ProtectedRoute>
        }>
          <Route index element={<Navigate to="/dashboard" />} />
          <Route path="dashboard" element={<Dashboard />} />
          
          {/* Full-featured pages */}
          <Route path="wallet" element={<Wallet />} />
          <Route path="loans" element={<Loans />} />
          <Route path="transactions" element={<Transactions />} />
          <Route path="users" element={<Users />} />
          <Route path="admin-stock" element={<AdminStock />} />
          <Route path="reports" element={<Reports />} />
          <Route path="analytics" element={<Analytics />} />
          <Route path="notifications" element={<Notifications />} />
          <Route path="settings" element={<Settings />} />
          <Route path="workflows" element={<Workflows />} />
          
          {/* Admin routes */}
          <Route path="admin/commissions" element={<CommissionManagement />} />
          <Route path="admin/cards" element={<CardManagement />} />
          <Route path="admin/virtual-cards" element={<VirtualCardPurchases />} />
          
          {/* Wallet sub-routes */}
          <Route path="wallets" element={<Wallets />} />
          <Route path="recharges" element={<Recharges />} />
          <Route path="transfers" element={<Transfers />} />
          <Route path="wallet/reload" element={<Wallet />} />
          <Route path="wallet/transfer" element={<Wallet />} />
          <Route path="nfc-payment" element={<Wallet />} />
          
          {/* Loans sub-routes */}
          <Route path="loans/active" element={<ActiveLoans />} />
          <Route path="loans/pending" element={<PendingLoans />} />
          <Route path="loans/repayments" element={<LoanRepayments />} />
          <Route path="loans/overdue" element={<ActiveLoans />} />
          <Route path="loans/apply" element={<Loans />} />
          <Route path="loans/history" element={<Loans />} />
          
          {/* Analytics sub-routes */}
          <Route path="analytics/nfc" element={<NFCAnalytics />} />
          <Route path="reports/cinetpay" element={<CinetPayReports />} />
          <Route path="monitoring" element={<Monitoring />} />
          
          {/* Nouvelles routes */}
          <Route path="countries" element={<Countries />} />
          <Route path="advertising" element={<Advertising />} />
          <Route path="stock-account" element={<StockAccount />} />
          <Route path="loans-management" element={<LoansManagement />} />
          <Route path="tim-business-recharges" element={<TIMBusinessRecharges />} />
          
          {/* Other sub-routes */}
          <Route path="flows" element={<Transactions />} />
        </Route>
        
        <Route path="*" element={<Navigate to="/dashboard" />} />
      </Routes>
    </Router>
  );
}

function App() {
  return (
    <AuthProvider>
      <AppContent />
    </AuthProvider>
  );
}

export default App;