#!/usr/bin/env node

/**
 * Script de build pour la production TIM CASH
 * Gère les variables d'environnement et la configuration
 */

const { execSync } = require('child_process');
const fs = require('fs');
const path = require('path');

console.log('🚀 Début du build de production pour TIM CASH...');

// Configuration par défaut
const defaultConfig = {
  VITE_API_URL: 'https://api.timcash.ci',
  VITE_APP_NAME: 'TIM CASH',
  VITE_APP_VERSION: '1.0.0',
  VITE_APP_ENV: 'production'
};

// Charger la configuration depuis les arguments ou les variables d'environnement
const config = {
  VITE_API_URL: process.env.VITE_API_URL || process.argv.find(arg => arg.startsWith('--api-url='))?.split('=')[1] || defaultConfig.VITE_API_URL,
  VITE_APP_NAME: process.env.VITE_APP_NAME || defaultConfig.VITE_APP_NAME,
  VITE_APP_VERSION: process.env.VITE_APP_VERSION || defaultConfig.VITE_APP_VERSION,
  VITE_APP_ENV: process.env.VITE_APP_ENV || defaultConfig.VITE_APP_ENV
};

console.log('📋 Configuration de production:');
console.log(`   API URL: ${config.VITE_API_URL}`);
console.log(`   App Name: ${config.VITE_APP_NAME}`);
console.log(`   Version: ${config.VITE_APP_VERSION}`);
console.log(`   Environment: ${config.VITE_APP_ENV}`);

// Créer le fichier .env.production
const envContent = Object.entries(config)
  .map(([key, value]) => `${key}=${value}`)
  .join('\n');

fs.writeFileSync('.env.production', envContent);
console.log('✅ Fichier .env.production créé');

// Nettoyer le dossier dist
if (fs.existsSync('dist')) {
  fs.rmSync('dist', { recursive: true, force: true });
  console.log('🧹 Dossier dist nettoyé');
}

// Installer les dépendances
console.log('📦 Installation des dépendances...');
try {
  execSync('npm ci --production=false', { stdio: 'inherit' });
  console.log('✅ Dépendances installées');
} catch (error) {
  console.error('❌ Erreur lors de l\'installation des dépendances:', error.message);
  process.exit(1);
}

// Build de production
console.log('🏗️ Build de production en cours...');
try {
  execSync('npm run build', { stdio: 'inherit' });
  console.log('✅ Build terminé avec succès');
} catch (error) {
  console.error('❌ Erreur lors du build:', error.message);
  process.exit(1);
}

// Vérifier que le build a réussi
if (!fs.existsSync('dist')) {
  console.error('❌ Le dossier dist n\'a pas été créé');
  process.exit(1);
}

// Copier les fichiers de configuration si nécessaire
const configFiles = [
  'env.production.example'
];

configFiles.forEach(file => {
  if (fs.existsSync(file)) {
    const destPath = path.join('dist', file);
    fs.copyFileSync(file, destPath);
    console.log(`📄 Fichier ${file} copié vers dist/`);
  }
});

// Afficher les informations de build
const distSize = getDirectorySize('dist');
console.log(`📊 Taille du build: ${distSize} MB`);

console.log('');
console.log('🎉 Build de production terminé avec succès!');
console.log('📁 Dossier de déploiement: ./dist');
console.log('');
console.log('📋 Prochaines étapes:');
console.log('   1. Vérifier que l\'API backend est accessible à:', config.VITE_API_URL);
console.log('   2. Configurer le serveur web pour servir les fichiers statiques');
console.log('   3. Configurer les en-têtes de sécurité (CSP)');
console.log('   4. Tester la connexion en production');

function getDirectorySize(dirPath) {
  let totalSize = 0;
  
  function calculateSize(itemPath) {
    const stats = fs.statSync(itemPath);
    if (stats.isDirectory()) {
      const files = fs.readdirSync(itemPath);
      files.forEach(file => {
        calculateSize(path.join(itemPath, file));
      });
    } else {
      totalSize += stats.size;
    }
  }
  
  calculateSize(dirPath);
  return (totalSize / (1024 * 1024)).toFixed(2);
}
