# 🔐 Gestion de Session - Déconnexion Automatique

## ✅ Fonctionnalité implémentée

### Déconnexion automatique après 10 minutes d'inactivité

Le système déconnecte automatiquement l'utilisateur après **10 minutes sans activité** pour des raisons de sécurité.

---

## ⏰ Comment ça fonctionne

### 1. **Détection d'activité**

Le système surveille les événements suivants :
- ✅ **Clics de souris** (`mousedown`, `click`)
- ✅ **Saisie clavier** (`keydown`)
- ✅ **Défilement** (`scroll`)
- ✅ **Toucher écran** (`touchstart`)

### 2. **Timer d'inactivité**

```typescript
const SESSION_TIMEOUT = 10 * 60 * 1000; // 10 minutes en millisecondes
```

- Chaque activité **réinitialise le timer**
- Si aucune activité pendant 10 minutes → **déconnexion automatique**
- Un timestamp est stocké dans `localStorage` : `lastActivity`

### 3. **Vérification au chargement**

Quand l'utilisateur recharge la page :
- ✅ Le système vérifie le `lastActivity`
- ✅ Si > 10 minutes → déconnexion immédiate
- ✅ Sinon → session restaurée

---

## 📁 Fichiers modifiés

### 1. **AuthContext.tsx**

```typescript
// Gestion de la session
const SESSION_TIMEOUT = 10 * 60 * 1000; // 10 minutes
let inactivityTimer: NodeJS.Timeout | null = null;

// Réinitialiser le timer à chaque activité
const resetInactivityTimer = () => {
  if (inactivityTimer) {
    clearTimeout(inactivityTimer);
  }

  if (state.isAuthenticated) {
    inactivityTimer = setTimeout(() => {
      console.log('⏰ Session expirée après 10 minutes d\'inactivité');
      logout();
      alert('Votre session a expiré après 10 minutes d\'inactivité. Veuillez vous reconnecter.');
    }, SESSION_TIMEOUT);

    localStorage.setItem('lastActivity', Date.now().toString());
  }
};

// Écouter les événements d'activité
useEffect(() => {
  if (!state.isAuthenticated) return;

  const events = ['mousedown', 'keydown', 'scroll', 'touchstart', 'click'];
  
  const handleActivity = () => {
    resetInactivityTimer();
  };

  events.forEach(event => {
    window.addEventListener(event, handleActivity);
  });

  resetInactivityTimer();

  return () => {
    events.forEach(event => {
      window.removeEventListener(event, handleActivity);
    });
    if (inactivityTimer) {
      clearTimeout(inactivityTimer);
    }
  };
}, [state.isAuthenticated]);
```

### 2. **SessionTimer.tsx** (Nouveau composant)

Composant visuel pour afficher le temps restant et alerter l'utilisateur.

```typescript
import SessionTimer from '../components/SessionTimer';

// Dans votre Layout ou Dashboard
<SessionTimer showWarning={true} />
```

**Fonctionnalités** :
- ⏰ Affiche le temps restant de la session
- ⚠️ Alerte 2 minutes avant expiration
- 🔄 Bouton pour prolonger la session
- 🎨 Design avec thème marron doré

---

## 🎯 Comportement détaillé

### Scénario 1 : Utilisateur actif

```
00:00 → Connexion
00:05 → Clic (timer réinitialisé)
02:30 → Scroll (timer réinitialisé)
05:00 → Saisie clavier (timer réinitialisé)
...
→ Session reste active tant qu'il y a de l'activité
```

### Scénario 2 : Utilisateur inactif

```
00:00 → Connexion
00:05 → Dernière activité
08:00 → Alerte "Session bientôt expirée" (2 min avant)
10:05 → DÉCONNEXION AUTOMATIQUE
→ Message : "Votre session a expiré après 10 minutes d'inactivité"
```

### Scénario 3 : Rechargement de page

```
00:00 → Connexion
05:00 → Fermeture de l'onglet
05:03 → Réouverture (< 10 min)
→ ✅ Session restaurée

00:00 → Connexion
05:00 → Fermeture de l'onglet
15:30 → Réouverture (> 10 min)
→ ❌ Déconnexion automatique
→ Message : "Votre session a expiré"
```

---

## 🎨 Intégration du composant SessionTimer

### Dans le Layout principal

```typescript
// src/components/Layout/Layout.tsx
import SessionTimer from '../SessionTimer';

const Layout: React.FC = () => {
  return (
    <div>
      <Header>
        {/* Afficher le timer dans le header */}
        <SessionTimer showWarning={true} />
      </Header>
      
      <Sidebar />
      <MainContent />
    </div>
  );
};
```

### Dans le Dashboard

```typescript
// src/pages/Dashboard/Dashboard.tsx
import SessionTimer from '../../components/SessionTimer';

const Dashboard: React.FC = () => {
  return (
    <div>
      {/* Timer visible dans le dashboard */}
      <div className="flex justify-between items-center">
        <h1>Dashboard</h1>
        <SessionTimer showWarning={true} />
      </div>
      
      {/* Reste du contenu */}
    </div>
  );
};
```

---

## 🔧 Configuration

### Modifier le délai d'expiration

Dans `AuthContext.tsx` :

```typescript
// Changer de 10 minutes à 30 minutes
const SESSION_TIMEOUT = 30 * 60 * 1000;

// Ou 5 minutes pour plus de sécurité
const SESSION_TIMEOUT = 5 * 60 * 1000;
```

### Modifier le seuil d'alerte

Dans `SessionTimer.tsx` :

```typescript
// Alerter 5 minutes avant au lieu de 2 minutes
const WARNING_THRESHOLD = 5 * 60 * 1000;
```

### Désactiver l'alerte visuelle

```typescript
<SessionTimer showWarning={false} />
```

---

## 📊 Données stockées

### localStorage

```javascript
{
  "token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
  "user": "{\"id\":\"123\",\"email\":\"user@example.com\",...}",
  "lastActivity": "1729890123456"  // Timestamp en millisecondes
}
```

### Nettoyage à la déconnexion

```typescript
localStorage.removeItem('token');
localStorage.removeItem('user');
localStorage.removeItem('lastActivity');
```

---

## 🎨 Apparence du SessionTimer

### Timer normal (> 2 minutes restantes)

```
┌─────────────────────┐
│ 🕐 Session: 8:45    │
└─────────────────────┘
```

### Alerte (< 2 minutes restantes)

```
┌─────────────────────────────────────────────┐
│ ⚠️ Session bientôt expirée                  │
│                                              │
│ Votre session expirera dans 1:30.           │
│ Cliquez pour prolonger.                     │
│                                              │
│ [Prolonger la session] [Fermer]             │
└─────────────────────────────────────────────┘
```

---

## 🔒 Sécurité

### Avantages

✅ **Protection contre l'accès non autorisé**
- Si l'utilisateur laisse sa session ouverte, elle expire automatiquement

✅ **Conformité RGPD**
- Limite l'exposition des données personnelles

✅ **Meilleure gestion des ressources**
- Libère les sessions inactives côté serveur

### Bonnes pratiques

1. ✅ **Ne pas stocker de données sensibles** dans localStorage
2. ✅ **Utiliser HTTPS** en production
3. ✅ **Implémenter un refresh token** pour les sessions longues
4. ✅ **Logger les déconnexions** pour l'audit

---

## 🧪 Tests

### Test 1 : Déconnexion après inactivité

```
1. Se connecter
2. Ne rien faire pendant 10 minutes
3. Vérifier la déconnexion automatique
4. Vérifier le message d'alerte
```

### Test 2 : Réinitialisation du timer

```
1. Se connecter
2. Attendre 9 minutes
3. Cliquer quelque part
4. Vérifier que le timer est réinitialisé
5. Attendre encore 9 minutes
6. Vérifier que la session est toujours active
```

### Test 3 : Rechargement de page

```
1. Se connecter
2. Attendre 5 minutes
3. Recharger la page (F5)
4. Vérifier que la session est restaurée
5. Attendre 6 minutes supplémentaires
6. Recharger la page
7. Vérifier la déconnexion (> 10 min total)
```

### Test 4 : Alerte d'expiration

```
1. Se connecter
2. Attendre 8 minutes
3. Vérifier l'apparition de l'alerte
4. Cliquer sur "Prolonger la session"
5. Vérifier que l'alerte disparaît
6. Vérifier que le timer est réinitialisé
```

---

## 📱 Compatibilité

### Navigateurs supportés

✅ Chrome / Edge (Chromium)
✅ Firefox
✅ Safari
✅ Opera

### Événements détectés

| Événement | Desktop | Mobile | Tablette |
|-----------|---------|--------|----------|
| `mousedown` | ✅ | ❌ | ❌ |
| `keydown` | ✅ | ✅ | ✅ |
| `scroll` | ✅ | ✅ | ✅ |
| `touchstart` | ❌ | ✅ | ✅ |
| `click` | ✅ | ✅ | ✅ |

---

## 🚀 Prochaines améliorations

### Fonctionnalités futures

1. **Refresh token automatique**
   - Prolonger la session sans redemander les identifiants

2. **Synchronisation multi-onglets**
   - Si l'utilisateur est actif dans un onglet, prolonger dans tous

3. **Historique des sessions**
   - Logger les connexions/déconnexions

4. **Notification push**
   - Alerter l'utilisateur même s'il n'est pas sur l'onglet

5. **Mode "Se souvenir de moi"**
   - Session de 30 jours avec refresh token

---

## ✅ Checklist d'implémentation

- [x] Modifier `AuthContext.tsx`
- [x] Créer `SessionTimer.tsx`
- [x] Ajouter la détection d'activité
- [x] Implémenter le timer d'inactivité
- [x] Vérifier la session au chargement
- [x] Nettoyer le localStorage à la déconnexion
- [x] Créer la documentation
- [ ] Intégrer SessionTimer dans le Layout
- [ ] Tester tous les scénarios
- [ ] Déployer en production

---

## 🎉 Résultat

✅ **Session sécurisée avec déconnexion automatique après 10 minutes d'inactivité**

✅ **Timer visuel pour informer l'utilisateur**

✅ **Alerte avant expiration**

✅ **Vérification au rechargement de page**

✅ **Nettoyage complet à la déconnexion**

La gestion de session est maintenant **complète et sécurisée** ! 🔐
