# 🚀 Guide de Déploiement en Production - TimCash

## 📋 Vue d'Ensemble

Ce guide détaille les étapes pour déployer TimCash en production de manière sécurisée et performante.

## 🛠️ Prérequis de Production

### Infrastructure Minimale
- **Serveur** : 2 CPU, 4GB RAM, 50GB SSD
- **OS** : Ubuntu 20.04+ ou CentOS 8+
- **Docker** : Version 20.10+
- **Docker Compose** : Version 2.0+
- **Domaine** : Certificat SSL valide
- **Base de données** : PostgreSQL 14+ ou MySQL 8+
- **Cache** : Redis 6+

### Services Externes Requis
- **CinetPay** : Compte marchand avec API keys
- **SMTP** : Service email (SendGrid, Mailgun, etc.)
- **Monitoring** : Sentry pour le tracking d'erreurs
- **CDN** : Cloudflare ou AWS CloudFront (optionnel)

## 🔧 Configuration de Production

### 1. Préparation du Serveur

```bash
# Mise à jour du système
sudo apt update && sudo apt upgrade -y

# Installation Docker
curl -fsSL https://get.docker.com -o get-docker.sh
sudo sh get-docker.sh
sudo usermod -aG docker $USER

# Installation Docker Compose
sudo curl -L "https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
sudo chmod +x /usr/local/bin/docker-compose

# Création des répertoires
sudo mkdir -p /opt/timcash
sudo chown $USER:$USER /opt/timcash
cd /opt/timcash
```

### 2. Configuration des Variables d'Environnement

```bash
# Créer le fichier .env de production
cp .env.example .env.production
```

**Fichier `.env.production`** :
```bash
# =============================================================================
# CONFIGURATION PRODUCTION TIMCASH
# =============================================================================

# Environnement
ENVIRONMENT=production
DEBUG=false

# Sécurité
SECRET_KEY=your-super-secure-secret-key-256-bits-minimum
JWT_ACCESS_TOKEN_EXPIRE_MINUTES=60    # 1 heure en production
JWT_REFRESH_TOKEN_EXPIRE_DAYS=7       # 7 jours en production

# Domaine et URLs
DOMAIN=your-domain.com
BACKEND_URL=https://api.your-domain.com
FRONTEND_URL=https://your-domain.com

# Base de données PostgreSQL
DATABASE_URL=postgresql://timcash_user:secure_password@postgres:5432/timcash_prod
DB_HOST=postgres
DB_PORT=5432
DB_NAME=timcash_prod
DB_USER=timcash_user
DB_PASSWORD=your-secure-db-password

# Redis
REDIS_URL=redis://redis:6379/0
REDIS_PASSWORD=your-redis-password

# CinetPay (PRODUCTION)
CINETPAY_API_KEY=your-production-cinetpay-api-key
CINETPAY_SITE_ID=your-production-site-id
CINETPAY_SECRET_KEY=your-production-secret-key
CINETPAY_RETURN_URL=https://your-domain.com/payment/success
CINETPAY_NOTIFY_URL=https://api.your-domain.com/wallet-paiements/callback

# Email SMTP
SMTP_HOST=smtp.sendgrid.net
SMTP_PORT=587
SMTP_USER=apikey
SMTP_PASSWORD=your-sendgrid-api-key
SMTP_FROM=noreply@your-domain.com

# Monitoring
SENTRY_DSN=https://your-sentry-dsn@sentry.io/project-id
LOG_LEVEL=INFO

# Rate Limiting (Production - Plus strict)
RATE_LIMIT_LOGIN=3/minute
RATE_LIMIT_WALLET=5/minute
RATE_LIMIT_TRANSFER=3/minute
RATE_LIMIT_WITHDRAW=2/minute

# SSL/TLS
SSL_ENABLED=true
SSL_CERT_PATH=/etc/ssl/certs/your-domain.com.pem
SSL_KEY_PATH=/etc/ssl/private/your-domain.com.key

# CORS (Production domains only)
CORS_ORIGINS=https://your-domain.com,https://www.your-domain.com
```

### 3. Docker Compose de Production

**Fichier `docker-compose.prod.yml`** :
```yaml
version: '3.8'

services:
  # Base de données PostgreSQL
  postgres:
    image: postgres:15-alpine
    container_name: timcash_postgres
    restart: unless-stopped
    environment:
      POSTGRES_DB: ${DB_NAME}
      POSTGRES_USER: ${DB_USER}
      POSTGRES_PASSWORD: ${DB_PASSWORD}
      POSTGRES_INITDB_ARGS: "--auth-host=scram-sha-256"
    volumes:
      - postgres_data:/var/lib/postgresql/data
      - ./backups:/backups
    networks:
      - timcash_network
    healthcheck:
      test: ["CMD-SHELL", "pg_isready -U ${DB_USER} -d ${DB_NAME}"]
      interval: 30s
      timeout: 10s
      retries: 3

  # Cache Redis
  redis:
    image: redis:7-alpine
    container_name: timcash_redis
    restart: unless-stopped
    command: redis-server --requirepass ${REDIS_PASSWORD} --appendonly yes
    volumes:
      - redis_data:/data
    networks:
      - timcash_network
    healthcheck:
      test: ["CMD", "redis-cli", "--raw", "incr", "ping"]
      interval: 30s
      timeout: 10s
      retries: 3

  # Backend API
  backend:
    build:
      context: ./backend
      dockerfile: Dockerfile.prod
    container_name: timcash_backend
    restart: unless-stopped
    env_file:
      - .env.production
    volumes:
      - ./logs:/app/logs
      - ./uploads:/app/uploads
    networks:
      - timcash_network
    depends_on:
      postgres:
        condition: service_healthy
      redis:
        condition: service_healthy
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:8000/health"]
      interval: 30s
      timeout: 10s
      retries: 3

  # Frontend
  frontend:
    build:
      context: .
      dockerfile: Dockerfile.prod
      args:
        VITE_API_URL: ${BACKEND_URL}
    container_name: timcash_frontend
    restart: unless-stopped
    networks:
      - timcash_network
    depends_on:
      - backend

  # Reverse Proxy Nginx
  nginx:
    image: nginx:alpine
    container_name: timcash_nginx
    restart: unless-stopped
    ports:
      - "80:80"
      - "443:443"
    volumes:
      - ./nginx/nginx.conf:/etc/nginx/nginx.conf:ro
      - ./nginx/ssl:/etc/ssl:ro
      - ./logs/nginx:/var/log/nginx
    networks:
      - timcash_network
    depends_on:
      - frontend
      - backend

  # Monitoring avec Prometheus
  prometheus:
    image: prom/prometheus:latest
    container_name: timcash_prometheus
    restart: unless-stopped
    volumes:
      - ./monitoring/prometheus.yml:/etc/prometheus/prometheus.yml:ro
      - prometheus_data:/prometheus
    networks:
      - timcash_network
    command:
      - '--config.file=/etc/prometheus/prometheus.yml'
      - '--storage.tsdb.path=/prometheus'
      - '--web.console.libraries=/etc/prometheus/console_libraries'
      - '--web.console.templates=/etc/prometheus/consoles'

  # Visualisation avec Grafana
  grafana:
    image: grafana/grafana:latest
    container_name: timcash_grafana
    restart: unless-stopped
    environment:
      GF_SECURITY_ADMIN_PASSWORD: ${GRAFANA_ADMIN_PASSWORD}
    volumes:
      - grafana_data:/var/lib/grafana
      - ./monitoring/grafana/dashboards:/etc/grafana/provisioning/dashboards:ro
      - ./monitoring/grafana/datasources:/etc/grafana/provisioning/datasources:ro
    networks:
      - timcash_network

networks:
  timcash_network:
    driver: bridge

volumes:
  postgres_data:
    driver: local
  redis_data:
    driver: local
  prometheus_data:
    driver: local
  grafana_data:
    driver: local
```

### 4. Configuration Nginx

**Fichier `nginx/nginx.conf`** :
```nginx
events {
    worker_connections 1024;
}

http {
    include /etc/nginx/mime.types;
    default_type application/octet-stream;

    # Logging
    log_format main '$remote_addr - $remote_user [$time_local] "$request" '
                    '$status $body_bytes_sent "$http_referer" '
                    '"$http_user_agent" "$http_x_forwarded_for"';

    access_log /var/log/nginx/access.log main;
    error_log /var/log/nginx/error.log warn;

    # Gzip compression
    gzip on;
    gzip_vary on;
    gzip_min_length 1024;
    gzip_types text/plain text/css text/xml text/javascript application/javascript application/xml+rss application/json;

    # Rate limiting
    limit_req_zone $binary_remote_addr zone=api:10m rate=10r/s;
    limit_req_zone $binary_remote_addr zone=auth:10m rate=5r/s;

    # SSL configuration
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers ECDHE-RSA-AES256-GCM-SHA512:DHE-RSA-AES256-GCM-SHA512:ECDHE-RSA-AES256-GCM-SHA384:DHE-RSA-AES256-GCM-SHA384;
    ssl_prefer_server_ciphers off;
    ssl_session_cache shared:SSL:10m;
    ssl_session_timeout 10m;

    # Security headers
    add_header X-Frame-Options DENY;
    add_header X-Content-Type-Options nosniff;
    add_header X-XSS-Protection "1; mode=block";
    add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;

    # Frontend
    server {
        listen 80;
        server_name your-domain.com www.your-domain.com;
        return 301 https://$server_name$request_uri;
    }

    server {
        listen 443 ssl http2;
        server_name your-domain.com www.your-domain.com;

        ssl_certificate /etc/ssl/your-domain.com.pem;
        ssl_certificate_key /etc/ssl/your-domain.com.key;

        location / {
            proxy_pass http://frontend:3000;
            proxy_set_header Host $host;
            proxy_set_header X-Real-IP $remote_addr;
            proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
            proxy_set_header X-Forwarded-Proto $scheme;
        }
    }

    # Backend API
    server {
        listen 443 ssl http2;
        server_name api.your-domain.com;

        ssl_certificate /etc/ssl/your-domain.com.pem;
        ssl_certificate_key /etc/ssl/your-domain.com.key;

        # Rate limiting pour l'authentification
        location /auth/ {
            limit_req zone=auth burst=10 nodelay;
            proxy_pass http://backend:8000;
            proxy_set_header Host $host;
            proxy_set_header X-Real-IP $remote_addr;
            proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
            proxy_set_header X-Forwarded-Proto $scheme;
        }

        # Rate limiting général pour l'API
        location / {
            limit_req zone=api burst=20 nodelay;
            proxy_pass http://backend:8000;
            proxy_set_header Host $host;
            proxy_set_header X-Real-IP $remote_addr;
            proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
            proxy_set_header X-Forwarded-Proto $scheme;
        }
    }
}
```

## 🚀 Procédure de Déploiement

### 1. Déploiement Initial

```bash
# 1. Cloner le projet
git clone https://github.com/your-org/timcash.git /opt/timcash
cd /opt/timcash

# 2. Configurer l'environnement
cp .env.example .env.production
# Éditer .env.production avec vos valeurs

# 3. Créer les certificats SSL
sudo certbot certonly --standalone -d your-domain.com -d www.your-domain.com -d api.your-domain.com

# 4. Copier les certificats
sudo cp /etc/letsencrypt/live/your-domain.com/fullchain.pem nginx/ssl/your-domain.com.pem
sudo cp /etc/letsencrypt/live/your-domain.com/privkey.pem nginx/ssl/your-domain.com.key

# 5. Créer les répertoires nécessaires
mkdir -p logs backups uploads monitoring

# 6. Construire et démarrer
docker-compose -f docker-compose.prod.yml up --build -d

# 7. Vérifier le déploiement
docker-compose -f docker-compose.prod.yml ps
docker-compose -f docker-compose.prod.yml logs -f
```

### 2. Migration de Base de Données

```bash
# Exécuter les migrations (si applicable)
docker-compose -f docker-compose.prod.yml exec backend python -m alembic upgrade head

# Créer les données initiales
docker-compose -f docker-compose.prod.yml exec backend python scripts/init_data.py
```

### 3. Tests Post-Déploiement

```bash
# Test de santé de l'API
curl https://api.your-domain.com/health

# Test de l'authentification
curl -X POST https://api.your-domain.com/auth/login \
  -H "Content-Type: application/json" \
  -d '{"email":"admin@timcash.com","password":"your-admin-password"}'

# Test du frontend
curl https://your-domain.com
```

## 🔒 Sécurité en Production

### 1. Firewall Configuration

```bash
# UFW (Ubuntu)
sudo ufw default deny incoming
sudo ufw default allow outgoing
sudo ufw allow ssh
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp
sudo ufw enable
```

### 2. Sauvegarde Automatisée

**Script `scripts/backup.sh`** :
```bash
#!/bin/bash
# Sauvegarde automatique TimCash

BACKUP_DIR="/opt/timcash/backups"
DATE=$(date +%Y%m%d_%H%M%S)

# Sauvegarde base de données
docker-compose -f docker-compose.prod.yml exec -T postgres pg_dump -U timcash_user timcash_prod > "$BACKUP_DIR/db_backup_$DATE.sql"

# Sauvegarde fichiers
tar -czf "$BACKUP_DIR/files_backup_$DATE.tar.gz" uploads logs

# Nettoyage (garder 30 jours)
find $BACKUP_DIR -name "*.sql" -mtime +30 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +30 -delete

echo "Sauvegarde terminée: $DATE"
```

**Crontab** :
```bash
# Sauvegarde quotidienne à 2h du matin
0 2 * * * /opt/timcash/scripts/backup.sh >> /var/log/timcash_backup.log 2>&1
```

### 3. Monitoring et Alertes

**Configuration Prometheus** (`monitoring/prometheus.yml`) :
```yaml
global:
  scrape_interval: 15s

scrape_configs:
  - job_name: 'timcash-backend'
    static_configs:
      - targets: ['backend:8000']
    metrics_path: /metrics
    scrape_interval: 30s

  - job_name: 'postgres'
    static_configs:
      - targets: ['postgres:5432']

  - job_name: 'redis'
    static_configs:
      - targets: ['redis:6379']

  - job_name: 'nginx'
    static_configs:
      - targets: ['nginx:80']
```

## 🔄 Mise à Jour en Production

### 1. Déploiement Blue-Green

```bash
# Script de mise à jour sans interruption
#!/bin/bash

# 1. Sauvegarder
./scripts/backup.sh

# 2. Récupérer la nouvelle version
git pull origin main

# 3. Construire les nouvelles images
docker-compose -f docker-compose.prod.yml build

# 4. Déploiement progressif
docker-compose -f docker-compose.prod.yml up -d --no-deps backend
docker-compose -f docker-compose.prod.yml up -d --no-deps frontend

# 5. Vérification
sleep 30
curl -f https://api.your-domain.com/health || exit 1

# 6. Nettoyage des anciennes images
docker image prune -f
```

### 2. Rollback Rapide

```bash
#!/bin/bash
# Rollback en cas de problème

# 1. Revenir à la version précédente
git checkout HEAD~1

# 2. Reconstruire et redéployer
docker-compose -f docker-compose.prod.yml up --build -d

# 3. Restaurer la base de données si nécessaire
# docker-compose -f docker-compose.prod.yml exec -T postgres psql -U timcash_user timcash_prod < backups/latest_backup.sql
```

## 📊 Monitoring et Maintenance

### 1. Dashboards Grafana

- **Performance API** : Temps de réponse, taux d'erreur
- **Base de Données** : Connexions, requêtes lentes
- **Infrastructure** : CPU, RAM, disque
- **Business** : Transactions, utilisateurs actifs

### 2. Alertes Critiques

- API indisponible > 1 minute
- Erreurs > 5% sur 5 minutes
- Disque > 85% plein
- Base de données inaccessible
- Certificat SSL expire dans 30 jours

### 3. Maintenance Régulière

**Hebdomadaire** :
- Vérification des logs d'erreur
- Mise à jour des dépendances de sécurité
- Test des sauvegardes

**Mensuelle** :
- Analyse des performances
- Nettoyage des logs anciens
- Révision des métriques business

**Trimestrielle** :
- Audit de sécurité complet
- Optimisation base de données
- Révision de la capacité

## 🆘 Plan de Reprise d'Activité

### 1. Procédures d'Urgence

**Panne Serveur** :
1. Basculer vers serveur de secours
2. Restaurer dernière sauvegarde
3. Rediriger le DNS
4. Communiquer aux utilisateurs

**Compromission Sécurité** :
1. Isoler le système
2. Changer tous les mots de passe/clés
3. Analyser les logs
4. Restaurer depuis sauvegarde saine
5. Notifier les autorités si nécessaire

### 2. Contacts d'Urgence

- **Équipe Technique** : +XXX-XXX-XXXX
- **Hébergeur** : Support 24/7
- **CinetPay** : Support technique
- **Sentry** : Monitoring automatique

---

**Version** : 1.0.0  
**Dernière Mise à Jour** : Septembre 2024  
**Statut** : ✅ Prêt pour Production
