# Guide d'Intégration Backend - Frontend TIM CASH

## 📋 Vue d'ensemble

Toutes les données du frontend ont été vidées et sont maintenant prêtes à être connectées au backend via des APIs REST.

## 🔧 Fichiers créés

### 1. Service API (`src/services/api.ts`)
Service centralisé contenant toutes les fonctions pour appeler les endpoints backend.

### 2. Hook personnalisé (`src/hooks/useApi.ts`)
Hook React pour faciliter les appels API avec gestion automatique du loading et des erreurs.

### 3. Documentation (`BACKEND_API_ENDPOINTS.md`)
Liste complète de tous les endpoints backend nécessaires avec leurs formats de requête/réponse.

---

## 🚀 Comment utiliser

### Exemple 1: Charger les utilisateurs dans la page Users

```typescript
import { useEffect, useState } from 'react';
import { getUsers } from '../../services/api';

const Users: React.FC = () => {
  const [users, setUsers] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    loadUsers();
  }, []);

  const loadUsers = async () => {
    try {
      setLoading(true);
      const data = await getUsers();
      setUsers(data.users);
    } catch (err) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  if (loading) return <div>Chargement...</div>;
  if (error) return <div>Erreur: {error}</div>;

  return (
    <div>
      {users.map(user => (
        <div key={user.id}>{user.full_name}</div>
      ))}
    </div>
  );
};
```

### Exemple 2: Avec le hook useApi

```typescript
import { useApi } from '../../hooks/useApi';
import { getUsers } from '../../services/api';

const Users: React.FC = () => {
  const { data, loading, error, refetch } = useApi(getUsers, { autoFetch: true });

  if (loading) return <div>Chargement...</div>;
  if (error) return <div>Erreur: {error.message}</div>;

  return (
    <div>
      <button onClick={refetch}>Actualiser</button>
      {data?.users.map(user => (
        <div key={user.id}>{user.full_name}</div>
      ))}
    </div>
  );
};
```

---

## 📊 Pages à connecter

### ✅ Pages déjà vidées (prêtes pour l'intégration)

1. **Dashboard** (`src/pages/Dashboard/Dashboard.tsx`)
   - API: `getDashboardStats()`
   - Afficher: statistiques, graphiques, transactions récentes

2. **Users** (`src/pages/Users/Users.tsx`)
   - API: `getUsers()`
   - Déjà partiellement connecté (fallback sur mock data)

3. **Wallets** (`src/pages/Wallets/Wallets.tsx`)
   - API: `getWallets()`
   - Afficher: liste des portefeuilles avec soldes

4. **Recharges** (`src/pages/Recharges/Recharges.tsx`)
   - API: `getRecharges()`
   - Afficher: historique des recharges CinetPay

5. **Flows** (`src/pages/Flows/Flows.tsx`)
   - API: `getTransactions()`
   - Afficher: toutes les transactions

6. **Active Loans** (`src/pages/Loans/ActiveLoans.tsx`)
   - API: `getLoans({ status: 'active' })`
   - Afficher: prêts en cours

7. **Pending Loans** (`src/pages/Loans/PendingLoans.tsx`)
   - API: `getLoans({ status: 'pending' })`
   - Actions: `approveLoan()`, `rejectLoan()`

8. **Loan Repayments** (`src/pages/Loans/LoanRepayments.tsx`)
   - API: `getLoanRepayments()`
   - Afficher: remboursements de prêts

9. **Admin Stock** (`src/pages/AdminStock/AdminStock.tsx`)
   - API: `getAdminStock()`, `updateAdminStock()`
   - Afficher: solde et historique

10. **Commissions** (`src/pages/Admin/CommissionManagement.tsx`)
    - API: `getCommissions()`, `updateCommission()`
    - Gérer: taux de commission

11. **Notifications** (`src/pages/Notifications/Notifications.tsx`)
    - API: `getNotifications()`, `markNotificationAsRead()`
    - Afficher: notifications système

12. **Reports** (`src/pages/Reports/Reports.tsx`)
    - API: `generateReport()`, `getReports()`
    - Générer et télécharger des rapports

13. **Analytics** (`src/pages/Analytics/Analytics.tsx`)
    - API: `getAnalytics()`
    - Afficher: graphiques et métriques

14. **NFC Analytics** (`src/pages/Analytics/NFCAnalytics.tsx`)
    - API: `getNFCAnalytics()`
    - Afficher: analytics NFC

---

## 🔐 Configuration

### 1. Variable d'environnement

Créer un fichier `.env` à la racine du projet frontend:

```env
VITE_API_URL=http://localhost:8001
```

### 2. Token JWT

Le token est automatiquement récupéré depuis `localStorage.getItem('token')` dans chaque requête.

---

## 📝 Étapes d'intégration par page

### Pour chaque page:

1. **Importer le service API**
   ```typescript
   import { getUsers } from '../../services/api';
   ```

2. **Remplacer les données statiques**
   ```typescript
   // Avant
   const users = [/* données statiques */];
   
   // Après
   const [users, setUsers] = useState([]);
   useEffect(() => {
     loadUsers();
   }, []);
   ```

3. **Ajouter la gestion du loading**
   ```typescript
   if (loading) return <LoadingSpinner />;
   ```

4. **Ajouter la gestion des erreurs**
   ```typescript
   if (error) return <ErrorMessage error={error} />;
   ```

5. **Tester avec le backend**
   - Démarrer le backend: `cd backend && uvicorn main:app --reload`
   - Démarrer le frontend: `npm run dev`
   - Vérifier dans la console du navigateur

---

## 🛠️ Backend requis

Le backend doit implémenter tous les endpoints listés dans `BACKEND_API_ENDPOINTS.md`.

### Endpoints prioritaires à implémenter en premier:

1. ✅ `/admin/users` (déjà implémenté)
2. 🔴 `/admin/dashboard/stats`
3. 🔴 `/admin/wallets`
4. 🔴 `/admin/transactions`
5. 🔴 `/admin/recharges`
6. 🔴 `/admin/loans`

---

## 🧪 Tests

### Tester une API:

```typescript
// Dans la console du navigateur
import { getUsers } from './services/api';
getUsers().then(console.log).catch(console.error);
```

---

## 📌 Notes importantes

1. **Toutes les données sont maintenant à zéro** dans le frontend
2. **Aucune donnée mock** ne s'affichera tant que le backend ne répond pas
3. **Les composants afficheront "Aucune donnée"** si le backend retourne des tableaux vides
4. **L'authentification** est requise pour tous les endpoints
5. **Les erreurs** sont catchées et affichées à l'utilisateur

---

## 🎯 Prochaines étapes

1. Implémenter les endpoints backend manquants
2. Connecter chaque page frontend une par une
3. Tester chaque intégration
4. Ajouter la pagination si nécessaire
5. Optimiser les performances (cache, debounce, etc.)

---

## 📞 Support

Si vous avez des questions sur l'intégration, référez-vous à:
- `BACKEND_API_ENDPOINTS.md` pour les formats d'API
- `src/services/api.ts` pour les fonctions disponibles
- `src/hooks/useApi.ts` pour l'utilisation du hook
