# 📡 Endpoints API TIM CASH

## ⚠️ **Important: Préfixe /api**

Tous les endpoints du router sont montés avec le préfixe `/api`.

---

## 🔐 **Authentification**

### **Login**
```
POST /api/financials/auth/user/login
```

**Request:**
```json
{
  "phone": "0769989178",
  "hashed_password": "votre-mot-de-passe"
}
```

**Response:**
```json
{
  "success": true,
  "data": {
    "access_token": "eyJ...",
    "refresh_token": "eyJ...",
    "token_type": "Bearer"
  }
}
```

---

### **Get Current User**
```
GET /api/auth/me
Authorization: Bearer {token}
```

**Response:**
```json
{
  "id": "uuid",
  "email": "admin@grouptimcash.com",
  "full_name": "Administrateur",
  "phone": "0769989178",
  "is_admin": true,
  "tim_account_type": "TIM_BUSINESS",
  "wallet": {
    "id": "uuid",
    "balance": 0,
    "max_balance": null,
    "currency": "FCFA",
    "is_active": true,
    "nfc_enabled": true
  },
  "created_at": "2025-10-25T..."
}
```

---

## 👥 **Gestion des utilisateurs**

### **Liste des utilisateurs (Admin)**
```
GET /admin/users
Authorization: Bearer {token}
```

**Response:**
```json
{
  "users": [
    {
      "id": "uuid",
      "username": "admin",
      "email": "admin@grouptimcash.com",
      "full_name": "Administrateur",
      "tim_account_type": "TIM_BUSINESS",
      "is_active": true,
      "wallet_balance": 0,
      "created_at": "2025-10-25T..."
    }
  ],
  "total_count": 1
}
```

---

## 💸 **Transactions**

### **Mes transactions**
```
GET /transactions
Authorization: Bearer {token}
```

**Query params:**
- `limit` (default: 50)
- `offset` (default: 0)

---

## 🌍 **Pays et villes**

### **Liste des pays**
```
GET /countries
```

### **Liste des villes**
```
GET /api/cities
```

---

## 📋 **Structure des URLs**

### **URLs complètes pour le frontend:**

```typescript
const API_BASE_URL = 'http://localhost:8001';

// Authentification
const LOGIN_URL = `${API_BASE_URL}/api/financials/auth/user/login`;
const ME_URL = `${API_BASE_URL}/api/auth/me`;

// Utilisateurs
const USERS_URL = `${API_BASE_URL}/admin/users`;

// Transactions
const TRANSACTIONS_URL = `${API_BASE_URL}/transactions`;

// Pays et villes
const COUNTRIES_URL = `${API_BASE_URL}/countries`;
const CITIES_URL = `${API_BASE_URL}/api/cities`;
```

---

## ⚠️ **Erreurs courantes**

### **404 Not Found**

**Problème:** URL sans le préfixe `/api`

❌ **Mauvais:**
```
POST /financials/auth/user/login
GET /auth/me
```

✅ **Correct:**
```
POST /api/financials/auth/user/login
GET /api/auth/me
```

---

### **401 Unauthorized**

**Causes possibles:**
1. Token manquant
2. Token invalide/expiré
3. Header Authorization mal formaté

✅ **Correct:**
```
Authorization: Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...
```

❌ **Incorrect:**
```
Authorization: eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...
```

---

## 🔍 **Tester les endpoints**

### **Avec curl:**

```powershell
# Login
curl -X POST http://localhost:8001/api/financials/auth/user/login `
  -H "Content-Type: application/json" `
  -d '{"phone":"0769989178","hashed_password":"PASSWORD"}'

# Get current user
curl http://localhost:8001/api/auth/me `
  -H "Authorization: Bearer TOKEN"

# Get users (admin)
curl http://localhost:8001/admin/users `
  -H "Authorization: Bearer TOKEN"
```

### **Avec fetch (JavaScript):**

```javascript
// Login
const loginResponse = await fetch('http://localhost:8001/api/financials/auth/user/login', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({
    phone: '0769989178',
    hashed_password: 'PASSWORD'
  })
});

const { data } = await loginResponse.json();
const token = data.access_token;

// Get current user
const meResponse = await fetch('http://localhost:8001/api/auth/me', {
  headers: {
    'Authorization': `Bearer ${token}`,
    'Content-Type': 'application/json'
  }
});
```

---

## 📝 **Notes importantes**

1. **Tous les endpoints du router** utilisent le préfixe `/api`
2. **Les endpoints définis directement dans main.py** (comme `/admin/users`) n'ont PAS le préfixe
3. **L'API utilise `phone` pour le login**, pas `email`
4. **Le token JWT expire après 30 minutes** par défaut

---

## ✅ **Checklist de connexion**

- [ ] Backend démarré sur le port 8001
- [ ] URL avec le préfixe `/api` pour les endpoints du router
- [ ] Header `Authorization: Bearer {token}` pour les routes protégées
- [ ] Token stocké dans localStorage après login
- [ ] Utilisation du `phone` pour le login (pas l'email)

---

**Date:** 25 octobre 2025  
**Version API:** 1.0  
**Base URL:** http://localhost:8001

---

**TIM CASH API** - Référence complète des endpoints 🚀
