import httpx
import json
import uuid
import hashlib
from datetime import datetime, timezone
from fastapi import HTTPException, Depends, Request
from sqlalchemy.orm import Session
from pydantic import BaseModel, Field
from typing import Optional, Dict, Any
import os
from enum import Enum

# Configuration CinetPay (identique à votre Flutter)
CINETPAY_API_KEY = os.getenv("CINETPAY_API_KEY", "")
CINETPAY_SITE_ID = "5880897"
CINETPAY_BASE_URL = "https://api-checkout.cinetpay.com/v2"
CINETPAY_NOTIFY_URL = "https://yourapp.com/api/cinetpay/notify"
CINETPAY_RETURN_URL = "https://yourapp.com/payment/success"

import logging

logger = logging.getLogger(__name__)

class CinetPayService:
    """Service backend pour supporter votre implémentation Flutter"""
    
    def __init__(self):
        self.api_key = CINETPAY_API_KEY
        self.site_id = CINETPAY_SITE_ID
        self.base_url = CINETPAY_BASE_URL
    
    def prepare_payment_data(self, transfer_data: Dict[str, Any]) -> Dict[str, Any]:
        """Prépare les données de paiement comme dans votre Flutter"""
        
        amount_int = int(transfer_data["amount"])
        
        # Utiliser les vraies données de l'utilisateur
        customer_name = transfer_data.get("customer_name", "TIMCASH")
        customer_surname = transfer_data.get("customer_surname", "USER")
        customer_email = transfer_data.get("customer_email", "noreply@timcash.ci")
        customer_id = transfer_data.get("customer_id", "172")
        
        return {
            'apikey': self.api_key,
            'site_id': self.site_id,
            'transaction_id': transfer_data["transaction_id"],
            'amount': amount_int,
            'currency': 'XOF',
            'alternative_currency': '',
            'description': transfer_data["description"],
            'customer_id': customer_id,
            'customer_name': customer_name,
            'customer_surname': customer_surname,
            'customer_email': customer_email,
            'customer_phone_number': transfer_data["recipient_phone"],
            'customer_address': transfer_data.get("customer_address", "Abidjan"),
            'customer_city': transfer_data.get("customer_city", "Abidjan"),
            'customer_country': 'CI',
            'customer_state': 'CI',
            'customer_zip_code': '00225',
            'notify_url': CINETPAY_NOTIFY_URL,
            'return_url': CINETPAY_RETURN_URL,
            'channels': 'ALL',
            'metadata': 'timcash_app',
            'lang': 'FR'
        }
    
    async def create_payment_url(self, transfer_data: Dict[str, Any]) -> Dict[str, Any]:
        """Crée l'URL de paiement CinetPay (logique similaire à Flutter)"""
        
        payment_data = self.prepare_payment_data(transfer_data)
        
        async with httpx.AsyncClient(timeout=30.0) as client:
            try:
                response = await client.post(
                    f"{self.base_url}/payment",
                    json=payment_data,
                    headers={
                        "Content-Type": "application/json",
                        "Accept": "application/json"
                    }
                )
                
                print(f"📡 Réponse API CinetPay: {response.status_code}")
                print(f"📡 Body: {response.text}")
                
                if response.status_code in [200, 201]:
                    response_data = response.json()
                    
                    if response_data.get('code') == '201':
                        return {
                            "success": True,
                            "payment_url": response_data['data']['payment_url'],
                            "cinetpay_token": response_data['data'].get('payment_token'),
                            "message": "URL de paiement créée avec succès"
                        }
                    else:
                        return {
                            "success": False,
                            "message": f"Erreur API CinetPay: {response_data.get('message', 'Erreur inconnue')}"
                        }
                else:
                    return {
                        "success": False,
                        "message": f"Erreur HTTP {response.status_code}: {response.text}"
                    }
                    
            except httpx.TimeoutException:
                return {
                    "success": False,
                    "message": "Timeout: Impossible de contacter CinetPay. Vérifiez votre connexion internet.",
                    "code": "NETWORK_TIMEOUT"
                }
            except Exception as e:
                return {
                    "success": False,
                    "message": "Erreur lors de la création du lien de paiement",
                    "code": "API_ERROR",
                    "details": str(e)
                }

# Instance du service
cinetpay_service = CinetPayService()
