#!/usr/bin/env python3
"""
Script interactif de configuration MySQL pour TIM CASH
Configure automatiquement la base de données et l'utilisateur
"""

import pymysql
import sys
import os
from getpass import getpass
from dotenv import load_dotenv, set_key

# Charger les variables d'environnement
load_dotenv()

def print_banner():
    """Afficher la bannière"""
    print("\n" + "=" * 60)
    print("🔧 Configuration MySQL pour TIM CASH")
    print("=" * 60 + "\n")

def get_root_credentials():
    """Obtenir les identifiants root MySQL"""
    print("📝 Connexion MySQL Root")
    print("-" * 60)
    
    host = input("Host MySQL [localhost]: ").strip() or "localhost"
    port = input("Port MySQL [3306]: ").strip() or "3306"
    
    try:
        port = int(port)
    except ValueError:
        print("❌ Port invalide, utilisation de 3306")
        port = 3306
    
    user = input("Utilisateur MySQL root [root]: ").strip() or "root"
    password = getpass("Mot de passe root MySQL: ")
    
    return {
        'host': host,
        'port': port,
        'user': user,
        'password': password,
        'charset': 'utf8mb4'
    }

def test_root_connection(config):
    """Tester la connexion root"""
    print("\n🔍 Test de connexion root...")
    
    try:
        connection = pymysql.connect(**config)
        cursor = connection.cursor()
        cursor.execute("SELECT VERSION()")
        version = cursor.fetchone()[0]
        
        print(f"✅ Connexion réussie!")
        print(f"   Version MySQL: {version}")
        
        connection.close()
        return True
        
    except pymysql.Error as e:
        print(f"❌ Échec de connexion: {e}")
        return False

def create_database_and_user(root_config):
    """Créer la base de données et l'utilisateur TIM CASH"""
    
    # Configuration par défaut
    db_name = os.getenv('MYSQL_DATABASE', 'timcash_db')
    db_user = os.getenv('MYSQL_USER', 'timcash')
    db_password = os.getenv('MYSQL_PASSWORD', 'timcash123')
    
    print("\n📊 Configuration de la base de données TIM CASH")
    print("-" * 60)
    print(f"Base de données: {db_name}")
    print(f"Utilisateur: {db_user}")
    print(f"Mot de passe: {db_password}")
    
    change = input("\nModifier ces valeurs? [o/N]: ").strip().lower()
    
    if change == 'o' or change == 'oui':
        db_name = input(f"Nom de la base [{db_name}]: ").strip() or db_name
        db_user = input(f"Utilisateur [{db_user}]: ").strip() or db_user
        db_password = getpass(f"Mot de passe [{db_password}]: ").strip() or db_password
    
    try:
        # Connexion root
        connection = pymysql.connect(**root_config)
        cursor = connection.cursor()
        
        # Créer la base de données
        print(f"\n🗄️ Création de la base de données '{db_name}'...")
        cursor.execute(f"DROP DATABASE IF EXISTS {db_name}")  # Nettoyer d'abord
        cursor.execute(f"CREATE DATABASE {db_name} CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci")
        print(f"✅ Base de données '{db_name}' créée")
        
        # Supprimer l'utilisateur s'il existe déjà
        print(f"\n👤 Configuration de l'utilisateur '{db_user}'...")
        try:
            cursor.execute(f"DROP USER IF EXISTS '{db_user}'@'localhost'")
            cursor.execute(f"DROP USER IF EXISTS '{db_user}'@'%'")
        except:
            pass
        
        # Créer l'utilisateur
        cursor.execute(f"CREATE USER '{db_user}'@'localhost' IDENTIFIED BY '{db_password}'")
        cursor.execute(f"CREATE USER '{db_user}'@'%' IDENTIFIED BY '{db_password}'")
        print(f"✅ Utilisateur '{db_user}' créé")
        
        # Accorder les privilèges
        print(f"\n🔑 Attribution des privilèges...")
        cursor.execute(f"GRANT ALL PRIVILEGES ON {db_name}.* TO '{db_user}'@'localhost'")
        cursor.execute(f"GRANT ALL PRIVILEGES ON {db_name}.* TO '{db_user}'@'%'")
        cursor.execute("FLUSH PRIVILEGES")
        print(f"✅ Privilèges accordés")
        
        connection.close()
        
        # Tester la connexion avec le nouvel utilisateur
        print(f"\n🔍 Test de connexion avec '{db_user}'...")
        test_config = {
            'host': root_config['host'],
            'port': root_config['port'],
            'user': db_user,
            'password': db_password,
            'database': db_name,
            'charset': 'utf8mb4'
        }
        
        test_conn = pymysql.connect(**test_config)
        test_cursor = test_conn.cursor()
        test_cursor.execute("SELECT 1")
        result = test_cursor.fetchone()
        
        if result[0] == 1:
            print("✅ Test de connexion réussi!")
        
        test_conn.close()
        
        # Mettre à jour le fichier .env
        print("\n📝 Mise à jour de la configuration...")
        
        env_file = '.env'
        if not os.path.exists(env_file):
            print("   Création du fichier .env...")
            with open('.env.mysql', 'r') as source:
                with open('.env', 'w') as dest:
                    dest.write(source.read())
        
        # URL de connexion
        database_url = f"mysql+pymysql://{db_user}:{db_password}@{root_config['host']}:{root_config['port']}/{db_name}"
        
        # Mettre à jour les variables
        set_key(env_file, 'DATABASE_URL', database_url)
        set_key(env_file, 'MYSQL_HOST', root_config['host'])
        set_key(env_file, 'MYSQL_PORT', str(root_config['port']))
        set_key(env_file, 'MYSQL_USER', db_user)
        set_key(env_file, 'MYSQL_PASSWORD', db_password)
        set_key(env_file, 'MYSQL_DATABASE', db_name)
        
        print(f"✅ Configuration sauvegardée dans {env_file}")
        print(f"\n🔗 URL de connexion:")
        print(f"   {database_url}")
        
        return True
        
    except pymysql.Error as e:
        print(f"\n❌ Erreur MySQL: {e}")
        return False
    except Exception as e:
        print(f"\n❌ Erreur: {e}")
        return False

def show_next_steps():
    """Afficher les prochaines étapes"""
    print("\n" + "=" * 60)
    print("🎉 Configuration MySQL terminée avec succès!")
    print("=" * 60)
    print("\n📋 Prochaines étapes:")
    print("   1. Vérifier l'état: python check_mysql_status.py")
    print("   2. Migrer les données: python migrate_sqlite_to_mysql.py")
    print("   3. Démarrer le serveur: python -m uvicorn main:app --reload --port 8001")
    print("\n💡 Documentation complète: MIGRATION_MYSQL_GUIDE.md")
    print("=" * 60 + "\n")

def main():
    """Fonction principale"""
    print_banner()
    
    # Vérifier PyMySQL
    try:
        import pymysql
    except ImportError:
        print("❌ PyMySQL non installé")
        print("💡 Installez avec: pip install pymysql")
        sys.exit(1)
    
    # Guide rapide
    print("Ce script va:")
    print("  1. Se connecter à MySQL avec les identifiants root")
    print("  2. Créer la base de données 'timcash_db'")
    print("  3. Créer l'utilisateur 'timcash'")
    print("  4. Configurer les privilèges")
    print("  5. Mettre à jour le fichier .env")
    
    print("\n⚠️  Assurez-vous que MySQL est installé et démarré!")
    input("\nAppuyez sur Entrée pour continuer ou Ctrl+C pour annuler...")
    
    # Obtenir les identifiants root
    root_config = get_root_credentials()
    
    # Tester la connexion root
    if not test_root_connection(root_config):
        print("\n❌ Impossible de se connecter à MySQL")
        print("💡 Vérifiez que:")
        print("   - MySQL est installé et démarré")
        print("   - Le mot de passe root est correct")
        print("   - Le port 3306 est accessible")
        sys.exit(1)
    
    # Créer la base et l'utilisateur
    if create_database_and_user(root_config):
        show_next_steps()
    else:
        print("\n❌ Échec de la configuration")
        sys.exit(1)

if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print("\n\n⚠️ Configuration annulée par l'utilisateur")
        sys.exit(0)
    except Exception as e:
        print(f"\n❌ Erreur inattendue: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)
