#!/usr/bin/env python3
"""
Script de configuration complète de la base de données TIM CASH
"""
import os
import sys
import subprocess
from pathlib import Path

def run_command(command, description, cwd=None):
    """Exécuter une commande avec gestion d'erreur"""
    print(f"\n🔄 {description}...")
    try:
        if cwd is None:
            cwd = os.getcwd()
        
        result = subprocess.run(command, shell=True, check=True, capture_output=True, text=True, cwd=cwd)
        print(f"✅ {description} - Succès")
        if result.stdout:
            print(f"   Output: {result.stdout.strip()}")
        return True
    except subprocess.CalledProcessError as e:
        print(f"❌ {description} - Erreur")
        if e.stderr:
            print(f"   Error: {e.stderr}")
        return False

def check_dependencies():
    """Vérifier les dépendances Python"""
    print("\n🔍 Vérification des dépendances...")
    
    required_packages = [
        'sqlalchemy',
        'alembic', 
        'python-dotenv',
        'bcrypt',
        'fastapi',
        'uvicorn'
    ]
    
    missing_packages = []
    
    for package in required_packages:
        try:
            __import__(package.replace('-', '_'))
            print(f"   ✅ {package}")
        except ImportError:
            print(f"   ❌ {package} - Manquant")
            missing_packages.append(package)
    
    if missing_packages:
        print(f"\n⚠️  Packages manquants: {', '.join(missing_packages)}")
        print("   Installez-les avec: pip install " + " ".join(missing_packages))
        return False
    
    print("✅ Toutes les dépendances sont présentes")
    return True

def setup_environment():
    """Configurer l'environnement"""
    print("\n🔧 Configuration de l'environnement...")
    
    # Créer le fichier .env s'il n'existe pas
    env_file = Path(".env")
    env_example = Path(".env.example")
    
    if not env_file.exists() and env_example.exists():
        print("   📄 Création du fichier .env depuis .env.example")
        env_file.write_text(env_example.read_text())
        print("   ✅ Fichier .env créé")
    elif env_file.exists():
        print("   ✅ Fichier .env existe déjà")
    else:
        print("   ⚠️  Aucun fichier .env ou .env.example trouvé")
    
    return True

def run_migrations():
    """Exécuter les migrations Alembic"""
    print("\n🗄️  Exécution des migrations...")
    
    # Vérifier si la base de données existe
    db_file = Path("timcash.db")
    if db_file.exists():
        print("   ✅ Base de données existe déjà")
    else:
        print("   📄 Création de la base de données...")
    
    # Exécuter les migrations
    success = run_command(
        "python -m alembic upgrade head",
        "Application des migrations Alembic"
    )
    
    return success

def create_admin_user():
    """Créer l'utilisateur admin"""
    print("\n👤 Création de l'utilisateur admin...")
    
    success = run_command(
        "python test_db.py",
        "Création de l'utilisateur admin et vérification de la base"
    )
    
    return success

def populate_sample_data():
    """Peupler avec des données d'exemple"""
    print("\n🌱 Peuplement avec des données d'exemple...")
    
    success = run_command(
        "python seed_data.py",
        "Insertion des données d'exemple TIM CASH"
    )
    
    return success

def verify_setup():
    """Vérifier que tout fonctionne"""
    print("\n✅ Vérification de la configuration...")
    
    try:
        # Importer et tester la base de données
        sys.path.append('.')
        from database import SessionLocal
        from models import User, Country, CommissionRate
        
        db = SessionLocal()
        
        # Compter les données
        user_count = db.query(User).count()
        country_count = db.query(Country).count()
        commission_count = db.query(CommissionRate).count()
        
        print(f"   👥 Utilisateurs: {user_count}")
        print(f"   🌍 Pays: {country_count}")
        print(f"   💰 Taux de commission: {commission_count}")
        
        db.close()
        
        if user_count > 0 and country_count > 0:
            print("✅ Base de données configurée correctement")
            return True
        else:
            print("❌ Données manquantes dans la base")
            return False
            
    except Exception as e:
        print(f"❌ Erreur lors de la vérification: {e}")
        return False

def main():
    """Fonction principale de configuration"""
    print("🚀 Configuration complète de la base de données TIM CASH")
    print("=" * 60)
    
    # Vérifier que nous sommes dans le bon répertoire
    if not Path("requirements.txt").exists():
        print("❌ Erreur: requirements.txt non trouvé. Exécutez ce script depuis le répertoire backend/")
        sys.exit(1)
    
    steps = [
        ("Vérification des dépendances", check_dependencies),
        ("Configuration de l'environnement", setup_environment),
        ("Exécution des migrations", run_migrations),
        ("Création de l'utilisateur admin", create_admin_user),
        ("Peuplement des données d'exemple", populate_sample_data),
        ("Vérification finale", verify_setup)
    ]
    
    success_count = 0
    for step_name, step_func in steps:
        if step_func():
            success_count += 1
        else:
            print(f"\n❌ Échec à l'étape: {step_name}")
            break
    
    print("\n" + "=" * 60)
    if success_count == len(steps):
        print("🎉 Configuration terminée avec succès!")
        print("\n📋 Informations importantes:")
        print("   • Base de données: timcash.db")
        print("   • Admin email: admin@timcash.ci")
        print("   • Admin password: admin123")
        print("   • Utilisateurs test: marie.kouassi@timcash.ci, jean.baptiste@timcash.ci")
        print("   • Mot de passe test: password123")
        print("\n🚀 Démarrer le serveur:")
        print("   python -m uvicorn main_db:app --reload --port 8001")
        print("\n🌐 API Documentation:")
        print("   http://localhost:8001/docs")
        print("\n⚠️  N'oubliez pas de changer les mots de passe en production!")
    else:
        print(f"❌ Configuration échouée ({success_count}/{len(steps)} étapes réussies)")
        sys.exit(1)

if __name__ == "__main__":
    main()
