from datetime import date, datetime, timedelta
import shutil
import uuid
from fastapi import File, HTTPException, UploadFile
from pydantic import BaseModel
from utility.stock import get_admin_stock
from models import Loan, LoanStatus, TimAccountType, Transaction, TransactionStatus, TransactionType, User
from schemas.loans import CommissionRequest, LoanRequestWithInfo, RepayLoanRequest
from services.users import get_user_wallet
from sqlalchemy.orm import Session
from utility.transactions import cinetpay_service 

async def create_loan_with_info(
    current_user: User,
    db: Session ,
    amount: float,
    recto: UploadFile = File(...),
    verso: UploadFile = File(...),
):
    # Vérifier les limites de prêt selon le type de compte
    user = db.query(User).filter(User.id == current_user).first()
    if not user:
        raise HTTPException(status_code=404, detail="Utilisateur introuvable")

    max_loan_amounts = {
        TimAccountType.TIM_MINI: 0,
        TimAccountType.TIM_MAXI: 10000,
        TimAccountType.TIM_BUSINESS: 50000
    }
    max_amount = max_loan_amounts.get(user.tim_account_type, 0)
    if amount > max_amount:
        raise HTTPException(status_code=400, detail=f"Loan amount exceeds limit of {max_amount} FCFA")

    # Vérifier s'il y a déjà un prêt actif
    active_loan = db.query(Loan).filter(
        Loan.user_id == user.id,
        Loan.status.in_([LoanStatus.PENDING, LoanStatus.APPROVED, LoanStatus.ACTIVE])
    ).first()
    if active_loan:
        raise HTTPException(status_code=400, detail="You already have an active loan")

    # Vérifier le stock admin
    admin_stock = get_admin_stock(db)
    if admin_stock.balance < amount:
        raise HTTPException(status_code=400, detail="Insufficient funds in admin stock")

    # Enregistrer les fichiers recto/verso
    recto_path = f"uploads/{uuid.uuid4()}_{recto.filename}"
    verso_path = f"uploads/{uuid.uuid4()}_{verso.filename}"
    with open(recto_path, "wb") as f:
        shutil.copyfileobj(recto.file, f)
    with open(verso_path, "wb") as f:
        shutil.copyfileobj(verso.file, f)

    # Créer le prêt avec informations utilisateur
    interest_rate = 0.06
    total_amount = amount * (1 + interest_rate)

    loan = Loan(
        id=str(uuid.uuid4()),
        user_id=user.id,
        amount=amount,
        interest_rate=interest_rate,
        penalty_rate=0.15,
        remaining_balance=total_amount,
        monthly_payment=total_amount,
        total_payments=1,
        payments_completed=0,
        status=LoanStatus.APPROVED,
        due_date=datetime.utcnow() + timedelta(days=3),
        days_overdue=0,
        document_recto=recto_path,  # Stocker le chemin du document recto
        document_verso=verso_path   # Stocker le chemin du document verso
    )
    db.add(loan)

    # Débiter le stock admin
    admin_stock.balance -= amount
    admin_stock.total_debits += amount

    # Créditer le portefeuille utilisateur
    wallet = get_user_wallet(current_user, db)
    wallet.balance += amount

    # Créer la transaction
    transaction = Transaction(
        id=str(uuid.uuid4()),
        user_id=user.id,
        transaction_type=TransactionType.PRET_MOBILE,
        amount=amount,
        commission_amount=amount * interest_rate,
        commission_rate=interest_rate,
        status=TransactionStatus.COMPLETED,
        description=f"Prêt SOS - {'Prêt d\'urgence'}",
        reference=f"LOAN-{loan.id[:8]}",
        external_reference=loan.id
    )
    db.add(transaction)
    db.commit()

    return {
        "loan_id": loan.id,
        "amount": loan.amount,
        "interest_amount": loan.amount * interest_rate,
        "total_repayment": total_amount,
        "due_date": loan.due_date,
        "status": "approved",
    }


async def create_commission_users(
    current_user: User,
    db: Session ,
    request: CommissionRequest,
):
    user = db.query(User).filter(User.id == current_user).first()
    if not user:
        raise HTTPException(status_code=404, detail="Utilisateur introuvable")
    # Créer la transaction
    transaction = Transaction(
        id=str(uuid.uuid4()),
        user_id=user.id,
        transaction_type=TransactionType.COMMISSION,
        amount=request.amount,
        status=TransactionStatus.PENDING,
        description=f"Activations de commission - {'Prêt d\'urgence'}",
        reference=f"COMM-{str(uuid.uuid4())[:8]}",

    )
    db.add(transaction)
    db.commit()
    
    payment_data = {
        "transaction_id": transaction.id,
        "amount": request.amount,  # ⚠️ montant final à payer
        "description": f"Paiment de commission ",
        "recipient_phone": user.phone
    }

        # Appeler CinetPay
    cinetpay_response = await cinetpay_service.create_payment_url(payment_data)
        


    return {
        "transaction_id": transaction.id,
        "description": f"Paiment de commission ",
        "payment_url": cinetpay_response["payment_url"],
    }




async def repay_loan(request: RepayLoanRequest, db: Session, user: User):
    # 1. Chercher le prêt
    loan = db.query(Loan).filter(Loan.id == request.loan_id, Loan.user_id == request.user_id).first()
    if not loan:
        raise HTTPException(status_code=404, detail="Prêt introuvable")
    if loan.status == "regle":
        raise HTTPException(status_code=400, detail="Le prêt est déjà réglé")

    # 2. Vérifier montant
    if request.amount > loan.remaining_balance:
        raise HTTPException(status_code=400, detail="Montant trop élevé")

    now = datetime.utcnow()
    penalty_amount = 0.0
    due_plus_3 = loan.due_date + timedelta(days=3)
    if now > due_plus_3:
        loan.days_overdue = (now - loan.due_date).days
        penalty_amount = loan.remaining_balance * loan.penalty_rate
        loan.remaining_balance += penalty_amount
        
    # 5. Créditer compte stock admin
    admin_stock = get_admin_stock(db)
    admin_stock.balance += request.amount
    admin_stock.total_credits += request.amount

    # 6. Mettre à jour prêt
    payment_amount = request.amount
    if penalty_amount > 0:
        payment_amount += penalty_amount  # Ajouter pénalité seulement si retard > 3 jours

    # 7. Créer transaction
    transaction = Transaction(
        id=str(uuid.uuid4()),
        user_id=request.user_id,
        amount=request.amount,
        status=TransactionStatus.PENDING,
        transaction_type = TransactionType.MOBILE_TRANSFER,
        reference=f"REPAY-{loan.id[:8]}"
    )
    db.add(transaction)
    db.commit()
    
    payment_data = {
    "transaction_id": transaction.id,
    "amount": payment_amount,  # ⚠️ montant final à payer
    "description": f"Remboursement prêt {loan.id}",
    "recipient_phone": user.phone
            }  # Appeler CinetPay
    cinetpay_response = await cinetpay_service.create_payment_url(payment_data)
        
    return {
        "message": "Remboursement effectué avec succès",
        "remaining_balance": loan.remaining_balance,
        "penalty_applied": penalty_amount,
        "days_overdue": loan.days_overdue,
        "loan_status": loan.status,
        "payment_url": cinetpay_response["payment_url"],
    }
