"""
Service de gestion des commissions TimCash
"""
from sqlalchemy.orm import Session
from models import CommissionRate, User
from typing import Dict, Optional
import uuid
from datetime import datetime

class CommissionService:
    """Service pour gérer les commissions configurables"""
    
    @staticmethod
    def get_commission_rates(db: Session) -> Dict[str, float]:
        """Récupérer les taux de commission actifs"""
        rates = db.query(CommissionRate).filter(
            CommissionRate.is_active == True
        ).all()
        
        commission_rates = {}
        for rate in rates:
            commission_rates[rate.category] = rate.value
            
        return commission_rates
    
    @staticmethod
    def get_recharge_commission_rates(db: Session) -> Dict[str, float]:
        """
        Récupérer les taux de commission pour les recharges

        IMPORTANT: CinetPay déduit automatiquement sa commission via leur API.
        On ne gère que la commission TIM CASH qui est ajoutée au montant.

        Returns:
            Dict avec:
            - timcash_rate: Taux de commission TIM CASH (appliqué par nous)
            - cinetpay_rate: Taux CinetPay (informatif, déduit automatiquement par leur API)
        """
        rates = db.query(CommissionRate).filter(
            CommissionRate.category == "recharge_cinetpay",
            CommissionRate.is_active == True
        ).all()

        if not rates:
            # Valeurs par défaut si aucune configuration
            return {
                "timcash_rate": 0.02,  # 2% TIM CASH (appliqué par nous)
                "cinetpay_rate": 0.01,  # 1% CinetPay (informatif, déduit par leur API)
            }

        # Prendre le premier taux actif
        rate = rates[0]
        # Le taux configuré est pour TIM CASH uniquement
        return {
            "timcash_rate": rate.value,  # Commission TIM CASH
            "cinetpay_rate": 0.01,  # CinetPay déduit automatiquement ~1%
        }
    
    @staticmethod
    def update_commission_rate(
        db: Session, 
        category: str, 
        new_rate: float, 
        updated_by: str,
        description: str = None
    ) -> CommissionRate:
        """Mettre à jour un taux de commission"""
        
        # Désactiver l'ancien taux
        old_rates = db.query(CommissionRate).filter(
            CommissionRate.category == category,
            CommissionRate.is_active == True
        ).all()
        
        for old_rate in old_rates:
            old_rate.is_active = False
            old_rate.updated_at = datetime.utcnow()
        
        # Créer un nouveau taux
        new_commission_rate = CommissionRate(
            id=str(uuid.uuid4()),
            name=f"Commission {category}",
            rate_type="percentage",
            value=new_rate,
            category=category,
            description=description or f"Taux de commission {new_rate*100}% pour {category}",
            is_active=True,
            updated_by=updated_by
        )
        
        db.add(new_commission_rate)
        db.commit()
        db.refresh(new_commission_rate)
        
        return new_commission_rate
    
    @staticmethod
    def initialize_default_commissions(db: Session, admin_user_id: str):
        """Initialiser les commissions par défaut"""
        
        default_commissions = [
            {
                "category": "recharge_cinetpay",
                "value": 0.03,  # 3%
                "description": "Commission sur recharges mobile money (1% CinetPay + 2% TimCash)"
            },
            {
                "category": "loan_interest",
                "value": 0.06,  # 6%
                "description": "Taux d'intérêt sur les prêts"
            },
            {
                "category": "loan_penalty",
                "value": 0.15,  # 15%
                "description": "Pénalité de retard sur les prêts"
            }
        ]
        
        for commission_data in default_commissions:
            # Vérifier si le taux existe déjà
            existing_rate = db.query(CommissionRate).filter(
                CommissionRate.category == commission_data["category"],
                CommissionRate.is_active == True
            ).first()
            
            if not existing_rate:
                commission_rate = CommissionRate(
                    id=str(uuid.uuid4()),
                    name=f"Commission {commission_data['category']}",
                    rate_type="percentage",
                    value=commission_data["value"],
                    category=commission_data["category"],
                    description=commission_data["description"],
                    is_active=True,
                    updated_by=admin_user_id
                )
                
                db.add(commission_rate)
        
        db.commit()
    
    @staticmethod
    def calculate_commissions(amount: float, commission_rates: Dict[str, float]) -> Dict[str, float]:
        """
        Calculer les commissions pour une recharge

        FLUX DE PAIEMENT:
        1. Client veut recharger X FCFA
        2. TIM CASH ajoute sa commission (ex: 2%) -> Montant envoyé à CinetPay
        3. CinetPay déduit automatiquement sa commission (~1%) via leur API
        4. Client reçoit X FCFA sur son wallet

        Args:
            amount: Montant que le client veut recevoir sur son wallet
            commission_rates: Taux de commission (timcash_rate, cinetpay_rate)

        Returns:
            Dict avec les montants calculés
        """
        # Commission TIM CASH (appliquée par nous)
        timcash_commission = amount * commission_rates.get("timcash_rate", 0.02)

        # Montant à envoyer à CinetPay (montant + commission TIM CASH)
        amount_to_cinetpay = amount + timcash_commission

        # Commission CinetPay (déduite automatiquement par leur API)
        # On l'estime pour information, mais c'est CinetPay qui la gère
        cinetpay_commission_estimate = amount_to_cinetpay * commission_rates.get("cinetpay_rate", 0.01)

        return {
            "net_amount": amount,  # Montant qui sera crédité au client
            "timcash_commission": round(timcash_commission, 2),  # Commission TIM CASH
            "amount_to_cinetpay": round(amount_to_cinetpay, 2),  # Montant envoyé à CinetPay
            "cinetpay_commission_estimate": round(cinetpay_commission_estimate, 2),  # Estimation (déduit par CinetPay)
            "total_payment": round(amount_to_cinetpay, 2),  # Montant total que le client paie
        }
