"""
Middleware pour bloquer les transactions des comptes bloqués
"""
from fastapi import HTTPException, status
from models import User

def check_account_blocked(user: User, allow_recharge: bool = False):
    """
    Vérifier si le compte est bloqué et bloquer les transactions si nécessaire
    
    Args:
        user: L'utilisateur
        allow_recharge: Si True, autorise les recharges même si le compte est bloqué
        
    Raises:
        HTTPException: Si le compte est bloqué et l'opération n'est pas autorisée
    """
    if user.is_blocked:
        if allow_recharge:
            # Autoriser uniquement les recharges pour rembourser le prêt
            return
        
        # Bloquer toutes les autres opérations
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail={
                "error": "ACCOUNT_BLOCKED",
                "message": "Votre compte est bloqué. Vous ne pouvez effectuer que des recharges pour rembourser votre prêt.",
                "reason": user.blocked_reason or "Prêt en retard",
                "action_required": "Rechargez votre wallet pour rembourser votre prêt et débloquer votre compte."
            }
        )

