#!/usr/bin/env python3
"""
Extension de la base de données SQLite TIM CASH existante
Ajoute toutes les nouvelles tables étendues à timcash.db
"""

import os
import sys
from datetime import datetime, timedelta
from sqlalchemy import create_engine, text, inspect
from sqlalchemy.orm import sessionmaker
import uuid
from dotenv import load_dotenv

# Charger les variables d'environnement
load_dotenv()

# Import des modèles au niveau module
try:
    from database import Base
    from models import *
    from extended_models import *
    print("✅ Imports des modèles réussis")
except Exception as e:
    print(f"❌ Erreur d'import: {e}")
    sys.exit(1)

def extend_sqlite_database():
    """Étendre la base SQLite existante avec les nouvelles tables"""
    print("🚀 TIM CASH - Extension de la base SQLite")
    print("=" * 50)
    
    # Configuration SQLite
    sqlite_url = "sqlite:///timcash.db"
    print(f"📂 Base de données: {sqlite_url}")
    
    # Vérifier si la base existe
    if not os.path.exists("timcash.db"):
        print("❌ Fichier timcash.db non trouvé !")
        print("💡 Créez d'abord la base avec: python recreate_database.py")
        return False
    
    try:
        # Créer le moteur SQLite
        engine = create_engine(sqlite_url, connect_args={"check_same_thread": False})
        
        # Vérifier les tables existantes
        print("🔍 Vérification des tables existantes...")
        inspector = inspect(engine)
        existing_tables = set(inspector.get_table_names())
        print(f"📋 Tables existantes: {len(existing_tables)}")
        for table in sorted(existing_tables):
            print(f"   ✅ {table}")
        
        # Créer toutes les tables (nouvelles + existantes)
        print("\n🏗️ Création/mise à jour des tables...")
        Base.metadata.create_all(bind=engine)
        
        # Vérifier les nouvelles tables
        inspector = inspect(engine)
        all_tables = set(inspector.get_table_names())
        new_tables = all_tables - existing_tables
        
        print(f"\n✅ {len(all_tables)} tables au total")
        
        if new_tables:
            print(f"🆕 {len(new_tables)} nouvelles tables ajoutées:")
            for table in sorted(new_tables):
                print(f"   ✨ {table}")
        else:
            print("ℹ️ Aucune nouvelle table ajoutée (toutes existent déjà)")
        
        # Organiser les tables par catégorie
        print(f"\n📊 Organisation des tables par catégorie:")
        
        categories = {
            "🔐 Authentification": ["users", "user_profiles", "kyc_documents"],
            "💰 Finance": ["wallets", "transactions", "loans", "commission_rates"],
            "🌍 Géographie": ["countries", "cities"],
            "📱 NFC": ["nfc_devices"],
            "🏪 Commerce": ["merchants", "products", "product_purchases", "merchant_reviews"],
            "🎯 Marketing": ["campaigns", "campaign_participations", "loyalty_programs", "referrals", "advertisements"],
            "🎧 Support": ["support_tickets", "support_messages"],
            "📊 Analytics": ["user_sessions", "transaction_analytics", "system_metrics"],
            "🔒 Sécurité": ["security_events", "audit_logs"],
            "⚙️ Configuration": ["system_configurations", "feature_flags"]
        }
        
        for category, category_tables in categories.items():
            print(f"\n{category}:")
            for table in category_tables:
                if table in all_tables:
                    status = "🆕" if table in new_tables else "✅"
                    print(f"   {status} {table}")
                else:
                    print(f"   ❌ {table} (manquante)")
        
        # Tables non catégorisées
        uncategorized = all_tables - set([table for cat_tables in categories.values() for table in cat_tables])
        if uncategorized:
            print(f"\n📋 Autres tables:")
            for table in sorted(uncategorized):
                status = "🆕" if table in new_tables else "✅"
                print(f"   {status} {table}")
        
        engine.dispose()
        return True
        
    except Exception as e:
        print(f"❌ Erreur lors de l'extension: {e}")
        import traceback
        traceback.print_exc()
        return False

def populate_extended_data():
    """Peupler les nouvelles tables avec des données de démonstration"""
    print("\n📦 Peuplement des nouvelles tables...")
    
    sqlite_url = "sqlite:///timcash.db"
    
    try:
        engine = create_engine(sqlite_url, connect_args={"check_same_thread": False})
        SessionLocal = sessionmaker(bind=engine)
        db = SessionLocal()
        
        # 1. Configurations système (si pas déjà présentes)
        print("⚙️ Ajout des configurations système...")
        system_configs = [
            {
                "id": "platform_name",
                "key": "platform_name",
                "value": "TIM CASH",
                "description": "Nom de la plateforme",
                "is_active": True
            },
            {
                "id": "default_currency", 
                "key": "default_currency",
                "value": "FCFA",
                "description": "Devise par défaut",
                "is_active": True
            },
            {
                "id": "max_loan_amount",
                "key": "max_loan_amount", 
                "value": "10000",
                "description": "Montant maximum des prêts SOS",
                "is_active": True
            },
            {
                "id": "nfc_daily_limit",
                "key": "nfc_daily_limit",
                "value": "500000", 
                "description": "Limite quotidienne NFC en FCFA",
                "is_active": True
            }
        ]
        
        for config_data in system_configs:
            existing = db.query(SystemConfiguration).filter_by(id=config_data["id"]).first()
            if not existing:
                config = SystemConfiguration(**config_data)
                db.add(config)
                print(f"   ✨ {config_data['key']}")
        
        # 2. Feature flags
        print("🚩 Ajout des feature flags...")
        feature_flags = [
            {
                "id": "nfc_payments_enabled",
                "name": "nfc_payments_enabled",
                "description": "Activer les paiements NFC",
                "is_enabled": True,
                "rollout_percentage": 100.0
            },
            {
                "id": "loan_sos_enabled",
                "name": "loan_sos_enabled", 
                "description": "Activer les prêts SOS",
                "is_enabled": True,
                "rollout_percentage": 100.0
            },
            {
                "id": "cinetpay_integration",
                "name": "cinetpay_integration",
                "description": "Intégration CinetPay",
                "is_enabled": True,
                "rollout_percentage": 100.0
            },
            {
                "id": "marketplace_enabled",
                "name": "marketplace_enabled",
                "description": "Activer la marketplace",
                "is_enabled": False,
                "rollout_percentage": 0.0
            }
        ]
        
        for flag_data in feature_flags:
            existing = db.query(FeatureFlag).filter_by(id=flag_data["id"]).first()
            if not existing:
                flag = FeatureFlag(**flag_data)
                db.add(flag)
                print(f"   🚩 {flag_data['name']}")
        
        # 3. Profils utilisateurs étendus (pour les utilisateurs existants)
        print("👤 Création des profils utilisateurs étendus...")
        users = db.query(User).all()
        for user in users:
            existing_profile = db.query(UserProfile).filter_by(user_id=user.id).first()
            if not existing_profile:
                profile = UserProfile(
                    id=f"profile_{user.id}",
                    user_id=user.id,
                    date_of_birth=datetime(1990, 1, 1).date(),
                    gender="other",
                    address="Abidjan, Côte d'Ivoire",
                    city="Abidjan",
                    postal_code="00225",
                    country="Côte d'Ivoire",
                    occupation="Utilisateur TIM CASH",
                    monthly_income=500000.0,  # 500K FCFA
                    kyc_status="pending",
                    kyc_level=1,
                    risk_score=50.0,
                    preferred_language="fr",
                    timezone="Africa/Abidjan"
                )
                db.add(profile)
                print(f"   👤 Profil pour {user.full_name}")
        
        db.commit()
        db.close()
        
        print("✅ Données étendues ajoutées avec succès")
        return True
        
    except Exception as e:
        print(f"❌ Erreur lors du peuplement: {e}")
        import traceback
        traceback.print_exc()
        return False

def main():
    """Fonction principale"""
    print("🚀 TIM CASH - Extension SQLite avec tables avancées")
    print("=" * 60)
    
    # Étape 1: Étendre la base avec les nouvelles tables
    if not extend_sqlite_database():
        print("\n💥 Échec de l'extension de la base")
        sys.exit(1)
    
    # Étape 2: Peupler avec des données de démonstration
    if not populate_extended_data():
        print("\n💥 Échec du peuplement des données")
        sys.exit(1)
    
    print("\n🎉 Extension SQLite complète réussie !")
    print(f"📊 Base de données: timcash.db ({os.path.getsize('timcash.db')} bytes)")
    print("\n📋 Prochaines étapes:")
    print("   1. Testez la configuration: python test_extended_tables.py")
    print("   2. Démarrez le serveur: python -m uvicorn main:app --reload --port 8001")
    print("   3. Accédez à l'API: http://localhost:8001/docs")

if __name__ == "__main__":
    main()
