"""
Gestionnaire WebSocket pour les notifications en temps réel
Permet de notifier l'application mobile des changements de configuration
"""

from typing import Dict, List, Set
from fastapi import WebSocket
import json
from datetime import datetime

class ConnectionManager:
    """Gestionnaire de connexions WebSocket"""
    
    def __init__(self):
        # Connexions actives par type de client
        self.active_connections: Dict[str, Set[WebSocket]] = {
            "mobile": set(),  # Applications mobiles
            "admin": set(),   # Interfaces admin
        }
        
        # Historique des dernières configurations
        self.config_versions: Dict[str, dict] = {}
    
    async def connect(self, websocket: WebSocket, client_type: str = "mobile"):
        """Accepter une nouvelle connexion WebSocket"""
        await websocket.accept()
        if client_type not in self.active_connections:
            self.active_connections[client_type] = set()
        self.active_connections[client_type].add(websocket)
        print(f"✅ Nouvelle connexion {client_type}: {len(self.active_connections[client_type])} clients connectés")
    
    def disconnect(self, websocket: WebSocket, client_type: str = "mobile"):
        """Déconnecter un client WebSocket"""
        if client_type in self.active_connections:
            self.active_connections[client_type].discard(websocket)
            print(f"❌ Déconnexion {client_type}: {len(self.active_connections[client_type])} clients restants")
    
    async def send_personal_message(self, message: str, websocket: WebSocket):
        """Envoyer un message à un client spécifique"""
        try:
            await websocket.send_text(message)
        except Exception as e:
            print(f"Erreur lors de l'envoi du message: {e}")
    
    async def broadcast_to_mobile(self, message: dict):
        """
        Diffuser un message à toutes les applications mobiles
        Utilisé pour notifier les changements de configuration
        """
        disconnected = set()
        
        for connection in self.active_connections.get("mobile", set()):
            try:
                await connection.send_json(message)
            except Exception as e:
                print(f"Erreur lors de la diffusion: {e}")
                disconnected.add(connection)
        
        # Nettoyer les connexions mortes
        for conn in disconnected:
            self.disconnect(conn, "mobile")
    
    async def notify_config_update(self, config_type: str, config_data: dict):
        """
        Notifier toutes les apps mobiles d'une mise à jour de configuration
        
        Args:
            config_type: Type de configuration (commission, advertisement, settings, etc.)
            config_data: Données de configuration mises à jour
        """
        # Enregistrer la version de la configuration
        version = datetime.now().isoformat()
        self.config_versions[config_type] = {
            "version": version,
            "data": config_data,
            "updated_at": version
        }
        
        # Créer le message de notification
        notification = {
            "type": "config_update",
            "config_type": config_type,
            "version": version,
            "data": config_data,
            "timestamp": version,
            "action": "reload"  # L'app mobile doit recharger cette config
        }
        
        # Diffuser aux applications mobiles
        await self.broadcast_to_mobile(notification)
        
        print(f"📢 Configuration '{config_type}' mise à jour et diffusée à {len(self.active_connections.get('mobile', set()))} clients mobiles")
    
    async def notify_advertisement_update(self, ad_data: dict, action: str = "create"):
        """
        Notifier d'une mise à jour de publicité
        
        Args:
            ad_data: Données de la publicité
            action: create, update, delete, activate, deactivate
        """
        notification = {
            "type": "advertisement_update",
            "action": action,
            "data": ad_data,
            "timestamp": datetime.now().isoformat()
        }
        
        await self.broadcast_to_mobile(notification)
        print(f"📢 Publicité {action}: diffusée à {len(self.active_connections.get('mobile', set()))} clients")
    
    async def notify_commission_update(self, commission_data: dict):
        """
        Notifier d'une mise à jour des taux de commission
        
        Args:
            commission_data: Nouveaux taux de commission
        """
        await self.notify_config_update("commissions", commission_data)
    
    async def notify_loan_settings_update(self, loan_settings: dict):
        """
        Notifier d'une mise à jour des paramètres de prêt
        
        Args:
            loan_settings: Nouveaux paramètres de prêt (montants, durées, taux)
        """
        await self.notify_config_update("loan_settings", loan_settings)
    
    async def notify_account_limits_update(self, limits_data: dict):
        """
        Notifier d'une mise à jour des limites de compte
        
        Args:
            limits_data: Nouvelles limites par type de compte
        """
        await self.notify_config_update("account_limits", limits_data)
    
    async def notify_nfc_settings_update(self, nfc_settings: dict):
        """
        Notifier d'une mise à jour des paramètres NFC
        
        Args:
            nfc_settings: Nouveaux paramètres NFC (limites, sécurité)
        """
        await self.notify_config_update("nfc_settings", nfc_settings)
    
    def get_config_version(self, config_type: str) -> dict:
        """Récupérer la dernière version d'une configuration"""
        return self.config_versions.get(config_type, {})
    
    def get_all_config_versions(self) -> dict:
        """Récupérer toutes les versions de configuration"""
        return {
            config_type: {
                "version": config_data.get("version"),
                "updated_at": config_data.get("updated_at")
            }
            for config_type, config_data in self.config_versions.items()
        }


# Instance globale du gestionnaire de connexions
manager = ConnectionManager()
