
from datetime import datetime, timedelta
import json
import logging
from utility.send_email import send_mail
from core.excption import NotFoundException
from models import User
from schemas.users import LoginResponse, LoginSchema, PreSignupRequest, CompleteSignupRequest, UserResponse, UserUpdate, VerifyOtpRequest
from fastapi import APIRouter, BackgroundTasks, Depends, File, HTTPException, Request, UploadFile, status
from core.config import settings
from services.users import get_user_wallet, login_user, pre_signup, verify_is_opt, complete_signup,update_user_profile
from database import get_db
from sqlalchemy.orm import Session
from fastapi.responses import JSONResponse
import logging
logger = logging.getLogger(__name__)

from frontnew.backend.utility.auth_token import create_access_token, create_refresh_token, get_current_user

router = APIRouter()

@router.post("/financials/auth/user/register-phone")
async def create_usrer(user: PreSignupRequest, db: Session = Depends(get_db), background_tasks: BackgroundTasks = BackgroundTasks()  ):
    return await pre_signup(user, db,background_tasks)


@router.post("/financials/auth/user/confirm-otp")
async def verify(request_data: VerifyOtpRequest, db: Session = Depends(get_db)):
    otp = request_data.otp
    logging.info(f"OTP reçu: {otp}")
    
    user = await verify_is_opt(db=db, request=request_data)
    if not user:
        raise HTTPException(status_code=400, detail="Invalid OTP or email")
     # Créer les tokens
    access_token_expires = timedelta(minutes=settings.ACCESS_TOKEN_EXPIRE_MINUTES)
    access_token = create_access_token(
        data={"sub": user.email}, expires_delta=access_token_expires
    )
    refresh_token = create_refresh_token(data={"sub": user.email})
    
    logger.info(f"New user registered: {user.email}")
    
    
    return {
        "success": 200,
        "data": {
           "access_token": access_token,
            "refresh_token": refresh_token,
            "token_type": "bearer"
            }
        }

@router.patch("/financials/auth/user/register-user")
def complete_user_signup(request:Request,payload: CompleteSignupRequest, db: Session = Depends(get_db)):
    logger.info(f"Login attempt at {datetime.utcnow().isoformat()} payload {payload} from {request.client.host}")
    return complete_signup(payload, db)

@router.patch("/financials/update/user/profile")
async def update_user_profiles(
    request: Request,
    file: UploadFile = File(...),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    logger.info(f"Login attempt at {datetime.utcnow().isoformat()} from {request.client.host}")
    return await update_user_profile(
        current_user=current_user,
        email= current_user.email,
        phone= current_user.phone,
        last_name= current_user.last_name,
        full_name= current_user.full_name,
        file=file,
        db=db
    )

@router.post("/financials/auth/user/login", response_model=LoginResponse)
async def login( resquest: Request, loginUser: LoginSchema  ,db: Session = Depends(get_db)):
    user_login = await login_user(loginUser, db)
    if not user_login:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="Invalid credentials")
    return user_login


@router.get("/auth/me")
async def get_current_user_info(
    request: Request,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    logger.info(f"Login attempt at {datetime.utcnow().isoformat()} from {request.client.host}")
    
    wallet = get_user_wallet(current_user.id, db)
    
    result = {
        "id": current_user.id,
        "username": current_user.username,
        "email": current_user.email,
        "full_name": current_user.full_name,
        "last_name": current_user.last_name,
        "phone": current_user.phone,
        "photo_profile": current_user.photo_profile,
        "is_admin": current_user.is_admin,
        "commission_payee": current_user.commission_payee,
        "tim_account_type": current_user.tim_account_type.value,
        "country": current_user.country.name,
        "wallet": {
            "id": wallet.id,
            "balance": wallet.balance,
            "max_balance": wallet.max_balance,
            "currency": wallet.currency,
            "is_active": wallet.is_active,
            "nfc_enabled": wallet.nfc_enabled
        },
        "created_at": current_user.created_at.isoformat() if current_user.created_at else None
    }

    # ✅ JSON UTF-8 correct
    json_data = json.dumps(result, ensure_ascii=False)

    # ✅ Réponse propre avec encodage UTF-8
    return JSONResponse(
        content=json.loads(json_data),
        media_type="application/json; charset=utf-8"
    )
    
@router.get("/users/by-phone/{phone}", response_model=UserResponse)
def get_user_by_phone(phone: str, db: Session = Depends(get_db)):
    user = db.query(User).filter(User.phone == phone).first()
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    return user



@router.post("/send-otp/")
async def send_otp(email: str, background_tasks: BackgroundTasks):
    from core.generate_otp import generate_otp
    otp = generate_otp()  # Générer un OTP dynamique
    background_tasks.add_task(send_mail, email, "Validation de compte", code=otp)
    return {"message": f"OTP envoyé à {email}"}

