"""
Routes API pour la gestion des transferts
"""
from fastapi import APIRouter, Depends, HTTPException, status, Query
from sqlalchemy.orm import Session
from sqlalchemy import or_, and_
from typing import List, Optional
from datetime import datetime, timedelta
import logging

from database import get_db
from models import User, Transaction, TransactionType, TransactionStatus
from frontnew.backend.utility.auth_token import get_current_user

router = APIRouter()
logger = logging.getLogger(__name__)

@router.get("/transfers")
async def get_user_transfers(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
    limit: int = Query(50, ge=1, le=100),
    offset: int = Query(0, ge=0),
    status: Optional[str] = Query(None),
    transfer_type: Optional[str] = Query(None),
    start_date: Optional[str] = Query(None),
    end_date: Optional[str] = Query(None)
):
    """
    Récupérer l'historique des transferts de l'utilisateur
    
    Filtres disponibles:
    - status: pending, completed, failed, cancelled
    - transfer_type: mobile_money_transfer, transfer_out, transfer_in, etc.
    - start_date: Date de début (format: YYYY-MM-DD)
    - end_date: Date de fin (format: YYYY-MM-DD)
    """
    try:
        logger.info(f"📊 Récupération des transferts pour {current_user.email}")
        
        # Query de base
        query = db.query(Transaction).filter(Transaction.user_id == current_user.id)
        
        # Filtrer par type de transfert
        if transfer_type:
            try:
                transaction_type = TransactionType[transfer_type.upper()]
                query = query.filter(Transaction.transaction_type == transaction_type)
            except KeyError:
                pass
        else:
            # Par défaut, afficher uniquement les transferts
            transfer_types = [
                TransactionType.MOBILE_MONEY_TRANSFER,
                TransactionType.TRANSFER_OUT,
                TransactionType.TRANSFER_IN,
                TransactionType.MOBILE_TRANSFER,
                TransactionType.VIREMENT_BANCAIRE
            ]
            query = query.filter(Transaction.transaction_type.in_(transfer_types))
        
        # Filtrer par statut
        if status:
            try:
                transaction_status = TransactionStatus[status.upper()]
                query = query.filter(Transaction.status == transaction_status)
            except KeyError:
                pass
        
        # Filtrer par date
        if start_date:
            try:
                start = datetime.fromisoformat(start_date)
                query = query.filter(Transaction.created_at >= start)
            except ValueError:
                pass
        
        if end_date:
            try:
                end = datetime.fromisoformat(end_date)
                # Ajouter 1 jour pour inclure toute la journée
                end = end + timedelta(days=1)
                query = query.filter(Transaction.created_at < end)
            except ValueError:
                pass
        
        # Compter le total
        total = query.count()
        
        # Pagination et tri
        transfers = query.order_by(Transaction.created_at.desc()).offset(offset).limit(limit).all()
        
        # Formater les résultats
        result = []
        for transfer in transfers:
            result.append({
                "id": transfer.id,
                "reference": transfer.reference,
                "type": transfer.transaction_type.value,
                "amount": transfer.amount,
                "commission_amount": transfer.commission_amount,
                "commission_rate": transfer.commission_rate,
                "status": transfer.status.value,
                "description": transfer.description,
                "merchant_name": transfer.merchant_name,
                "payment_method": transfer.payment_method.value if transfer.payment_method else None,
                "external_reference": transfer.external_reference,
                "created_at": transfer.created_at.isoformat() if transfer.created_at else None,
                "completed_at": transfer.completed_at.isoformat() if transfer.completed_at else None,
            })
        
        logger.info(f"✅ {len(result)} transferts récupérés")
        
        return {
            "success": True,
            "total": total,
            "limit": limit,
            "offset": offset,
            "transfers": result
        }
        
    except Exception as e:
        logger.error(f"❌ Erreur récupération transferts: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Erreur lors de la récupération des transferts: {str(e)}"
        )

@router.get("/transfers/{transfer_id}")
async def get_transfer_details(
    transfer_id: str,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """
    Récupérer les détails d'un transfert spécifique
    """
    try:
        transfer = db.query(Transaction).filter(
            Transaction.id == transfer_id,
            Transaction.user_id == current_user.id
        ).first()
        
        if not transfer:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="Transfert non trouvé"
            )
        
        return {
            "success": True,
            "transfer": {
                "id": transfer.id,
                "reference": transfer.reference,
                "type": transfer.transaction_type.value,
                "amount": transfer.amount,
                "commission_amount": transfer.commission_amount,
                "commission_rate": transfer.commission_rate,
                "status": transfer.status.value,
                "description": transfer.description,
                "merchant_name": transfer.merchant_name,
                "payment_method": transfer.payment_method.value if transfer.payment_method else None,
                "external_reference": transfer.external_reference,
                "location": transfer.location,
                "device_info": transfer.device_info,
                "created_at": transfer.created_at.isoformat() if transfer.created_at else None,
                "completed_at": transfer.completed_at.isoformat() if transfer.completed_at else None,
            }
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Erreur récupération détails transfert: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Erreur lors de la récupération: {str(e)}"
        )

@router.get("/transfers/stats/summary")
async def get_transfers_summary(
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db),
    period: str = Query("month", regex="^(week|month|year)$")
):
    """
    Récupérer un résumé des transferts (statistiques)
    """
    try:
        # Calculer la date de début selon la période
        now = datetime.utcnow()
        if period == "week":
            start_date = now - timedelta(days=7)
        elif period == "month":
            start_date = now - timedelta(days=30)
        else:  # year
            start_date = now - timedelta(days=365)
        
        # Récupérer tous les transferts de la période
        transfers = db.query(Transaction).filter(
            Transaction.user_id == current_user.id,
            Transaction.created_at >= start_date,
            Transaction.transaction_type.in_([
                TransactionType.MOBILE_MONEY_TRANSFER,
                TransactionType.TRANSFER_OUT,
                TransactionType.TRANSFER_IN,
                TransactionType.MOBILE_TRANSFER,
                TransactionType.VIREMENT_BANCAIRE
            ])
        ).all()
        
        # Calculer les statistiques
        total_transfers = len(transfers)
        total_amount_sent = sum(abs(t.amount) for t in transfers if t.amount < 0)
        total_amount_received = sum(t.amount for t in transfers if t.amount > 0)
        total_fees = sum(t.commission_amount for t in transfers)
        
        completed = len([t for t in transfers if t.status == TransactionStatus.COMPLETED])
        pending = len([t for t in transfers if t.status == TransactionStatus.PENDING])
        failed = len([t for t in transfers if t.status == TransactionStatus.FAILED])
        
        return {
            "success": True,
            "period": period,
            "summary": {
                "total_transfers": total_transfers,
                "total_amount_sent": total_amount_sent,
                "total_amount_received": total_amount_received,
                "total_fees": total_fees,
                "completed": completed,
                "pending": pending,
                "failed": failed
            }
        }
        
    except Exception as e:
        logger.error(f"❌ Erreur récupération statistiques: {str(e)}")
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Erreur lors de la récupération des statistiques: {str(e)}"
        )
