#!/usr/bin/env python3
"""
Script pour activer la Côte d'Ivoire et toutes ses villes
"""

from sqlalchemy import create_engine, text
from sqlalchemy.orm import sessionmaker
import os
from dotenv import load_dotenv

# Charger les variables d'environnement
load_dotenv()

# Configuration de la base de données
DATABASE_URL = os.getenv('DATABASE_URL', 'mysql+pymysql://root:@localhost:3306/timcash_db')

# Créer le moteur et la session
engine = create_engine(DATABASE_URL)
SessionLocal = sessionmaker(bind=engine)

def activate_cote_ivoire():
    """Active la Côte d'Ivoire et toutes ses villes"""
    session = SessionLocal()
    
    try:
        print("\n" + "="*60)
        print("ACTIVATION DE LA CÔTE D'IVOIRE")
        print("="*60)
        
        # Activer le pays
        update_country = text("""
            UPDATE countries 
            SET is_active = 1 
            WHERE code = 'CI'
        """)
        
        result = session.execute(update_country)
        session.commit()
        
        print(f"\n✅ Pays activé: Côte d'Ivoire")
        
        # Activer toutes les villes
        update_cities = text("""
            UPDATE cities 
            SET is_active = 1 
            WHERE country_id = (SELECT id FROM countries WHERE code = 'CI')
        """)
        
        result = session.execute(update_cities)
        session.commit()
        
        print(f"✅ Toutes les villes activées")
        
        # Vérifier
        verify_query = text("""
            SELECT 
                c.name as country_name,
                c.is_active as country_active,
                COUNT(ci.id) as total_cities,
                SUM(CASE WHEN ci.is_active = 1 THEN 1 ELSE 0 END) as active_cities
            FROM countries c
            LEFT JOIN cities ci ON c.id = ci.country_id
            WHERE c.code = 'CI'
            GROUP BY c.id, c.name, c.is_active
        """)
        
        result = session.execute(verify_query)
        row = result.fetchone()
        
        print("\n" + "="*60)
        print("✅ ACTIVATION TERMINÉE")
        print("="*60)
        if row:
            print(f"🇨🇮 Pays: {row[0]}")
            print(f"📊 Statut pays: {'Actif ✅' if row[1] else 'Inactif ❌'}")
            print(f"📍 Villes totales: {row[2]}")
            print(f"✅ Villes actives: {row[3]}")
        print("="*60)
        
    except Exception as e:
        session.rollback()
        print(f"\n❌ Erreur: {e}")
        raise
    finally:
        session.close()

if __name__ == "__main__":
    try:
        activate_cote_ivoire()
    except Exception as e:
        print(f"\n❌ Erreur: {e}")
