# API de Synchronisation des Cartes NFC - Backoffice ↔ Application Mobile

## 📋 Vue d'ensemble

Cette API permet de gérer les cartes NFC depuis le backoffice et de synchroniser automatiquement les changements avec l'application mobile. Toute carte créée, activée ou désactivée depuis le backoffice sera immédiatement reflétée dans l'application mobile.

---

## 🔐 Authentification

Tous les endpoints nécessitent une authentification admin avec un token JWT :

```http
Authorization: Bearer <admin_access_token>
```

---

## 📡 Endpoints Disponibles

### 1. **Créer une Carte NFC**

**Endpoint:** `POST /admin/nfc-cards/create`

**Description:** Créer une nouvelle carte NFC depuis le backoffice. La carte sera automatiquement synchronisée avec l'application mobile de l'utilisateur.

**Body:**
```json
{
  "card_uid": "NFC-UID-12345",
  "card_number": "CARD-87654321",
  "user_id": "uuid-de-l-utilisateur",
  "card_type": "physical",
  "max_balance": 50000.0,
  "is_primary": true
}
```

**Réponse (200 OK):**
```json
{
  "success": true,
  "message": "Carte NFC créée avec succès",
  "card": {
    "id": "uuid-de-la-carte",
    "card_uid": "NFC-UID-12345",
    "card_number": "CARD-87654321",
    "user_id": "uuid-de-l-utilisateur",
    "user_name": "Jean Dupont",
    "user_phone": "+225 07 12 34 56 78",
    "status": "active",
    "balance": 0.0,
    "max_balance": 50000.0,
    "is_primary": true,
    "created_at": "2025-11-05T12:00:00"
  }
}
```

**Erreurs:**
- `400 Bad Request` : UID déjà existant ou données invalides
- `401 Unauthorized` : Token invalide ou manquant
- `403 Forbidden` : Utilisateur non admin
- `500 Internal Server Error` : Erreur serveur

---

### 2. **Mettre à Jour le Statut d'une Carte**

**Endpoint:** `PUT /admin/nfc-cards/{card_id}/status`

**Description:** Activer, désactiver ou bloquer une carte NFC. Le changement sera automatiquement synchronisé avec l'application mobile.

**Statuts disponibles:**
- `active` : Carte active et utilisable
- `inactive` : Carte désactivée temporairement
- `blocked` : Carte bloquée (nécessite intervention admin)
- `lost` : Carte déclarée perdue
- `stolen` : Carte déclarée volée

**Body:**
```json
{
  "status": "inactive",
  "reason": "Carte désactivée à la demande du client"
}
```

**Réponse (200 OK):**
```json
{
  "success": true,
  "message": "Statut de la carte mis à jour: active → inactive",
  "card": {
    "id": "uuid-de-la-carte",
    "card_uid": "NFC-UID-12345",
    "card_number": "CARD-87654321",
    "user_id": "uuid-de-l-utilisateur",
    "user_name": "Jean Dupont",
    "user_phone": "+225 07 12 34 56 78",
    "status": "inactive",
    "balance": 0.0,
    "max_balance": 50000.0,
    "is_primary": true,
    "updated_at": "2025-11-05T12:15:00"
  },
  "change": {
    "old_status": "active",
    "new_status": "inactive",
    "reason": "Carte désactivée à la demande du client",
    "changed_by": "admin@timcash.ci",
    "changed_at": "2025-11-05T12:15:00"
  }
}
```

**Erreurs:**
- `400 Bad Request` : Statut invalide
- `404 Not Found` : Carte introuvable
- `401 Unauthorized` : Token invalide
- `403 Forbidden` : Utilisateur non admin

---

### 3. **Récupérer Toutes les Cartes NFC**

**Endpoint:** `GET /admin/nfc-cards`

**Description:** Récupérer la liste de toutes les cartes NFC avec pagination et filtres.

**Query Parameters:**
- `limit` (int, default: 100) : Nombre de cartes à retourner
- `offset` (int, default: 0) : Décalage pour la pagination
- `status` (string, optional) : Filtrer par statut (active, inactive, blocked, lost, stolen)
- `user_id` (string, optional) : Filtrer par utilisateur

**Exemples:**
```
GET /admin/nfc-cards?limit=50&offset=0
GET /admin/nfc-cards?status=active
GET /admin/nfc-cards?user_id=uuid-de-l-utilisateur
```

**Réponse (200 OK):**
```json
{
  "cards": [
    {
      "id": "uuid-de-la-carte",
      "card_uid": "NFC-UID-12345",
      "card_number": "CARD-87654321",
      "user_id": "uuid-de-l-utilisateur",
      "user_name": "Jean Dupont",
      "user_phone": "+225 07 12 34 56 78",
      "user_email": "jean.dupont@example.com",
      "status": "active",
      "balance": 0.0,
      "max_balance": 50000.0,
      "is_primary": true,
      "last_used_at": "2025-11-05T10:30:00",
      "created_at": "2025-11-01T08:00:00",
      "updated_at": "2025-11-05T12:15:00"
    }
  ],
  "total_count": 150,
  "limit": 50,
  "offset": 0
}
```

---

### 4. **Récupérer les Détails d'une Carte**

**Endpoint:** `GET /admin/nfc-cards/{card_id}`

**Description:** Récupérer les détails complets d'une carte NFC, incluant les informations utilisateur et les dernières transactions.

**Réponse (200 OK):**
```json
{
  "card": {
    "id": "uuid-de-la-carte",
    "card_uid": "NFC-UID-12345",
    "card_number": "CARD-87654321",
    "user_id": "uuid-de-l-utilisateur",
    "user_name": "Jean Dupont",
    "user_phone": "+225 07 12 34 56 78",
    "user_email": "jean.dupont@example.com",
    "status": "active",
    "balance": 0.0,
    "max_balance": 50000.0,
    "is_primary": true,
    "last_used_at": "2025-11-05T10:30:00",
    "created_at": "2025-11-01T08:00:00",
    "updated_at": "2025-11-05T12:15:00"
  },
  "user": {
    "id": "uuid-de-l-utilisateur",
    "full_name": "Jean Dupont",
    "phone": "+225 07 12 34 56 78",
    "email": "jean.dupont@example.com",
    "tim_account_type": "TIM_MAXI"
  },
  "recent_transactions": [
    {
      "id": "uuid-transaction",
      "operation_type": "read",
      "device_info": "Samsung Galaxy S21",
      "location": "Abidjan, Côte d'Ivoire",
      "created_at": "2025-11-05T10:30:00"
    }
  ]
}
```

**Erreurs:**
- `404 Not Found` : Carte introuvable

---

### 5. **Supprimer une Carte NFC**

**Endpoint:** `DELETE /admin/nfc-cards/{card_id}`

**Description:** Désactiver une carte NFC (soft delete). La carte ne sera pas supprimée de la base de données mais son statut sera changé à "inactive".

**Réponse (200 OK):**
```json
{
  "success": true,
  "message": "Carte NFC désactivée avec succès",
  "card_id": "uuid-de-la-carte"
}
```

**Erreurs:**
- `404 Not Found` : Carte introuvable

---

## 🔄 Synchronisation avec l'Application Mobile

### Mécanisme de Synchronisation

Lorsqu'une carte est créée, activée ou désactivée depuis le backoffice, les changements sont automatiquement synchronisés avec l'application mobile via :

1. **WebSocket (Temps Réel)** : Notification push instantanée à l'application mobile
2. **Pull Refresh** : L'application mobile peut rafraîchir la liste des cartes à tout moment
3. **Push Notifications** : Notification push pour informer l'utilisateur des changements

### Endpoints Mobile (À implémenter)

L'application mobile doit implémenter les endpoints suivants pour recevoir les mises à jour :

```
GET /api/nfc-cards/my-cards - Récupérer mes cartes
POST /api/nfc-cards/sync - Synchroniser les cartes
```

---

## 📊 Exemples d'Utilisation

### Exemple 1 : Créer une carte pour un utilisateur

```bash
curl -X POST http://localhost:8001/admin/nfc-cards/create \
  -H "Authorization: Bearer <admin_token>" \
  -H "Content-Type: application/json" \
  -d '{
    "card_uid": "NFC-001-2025",
    "card_number": "CARD-12345678",
    "user_id": "user-uuid-123",
    "card_type": "physical",
    "max_balance": 50000.0,
    "is_primary": true
  }'
```

### Exemple 2 : Désactiver une carte

```bash
curl -X PUT http://localhost:8001/admin/nfc-cards/card-uuid-123/status \
  -H "Authorization: Bearer <admin_token>" \
  -H "Content-Type: application/json" \
  -d '{
    "status": "inactive",
    "reason": "Carte perdue par le client"
  }'
```

### Exemple 3 : Réactiver une carte

```bash
curl -X PUT http://localhost:8001/admin/nfc-cards/card-uuid-123/status \
  -H "Authorization: Bearer <admin_token>" \
  -H "Content-Type: application/json" \
  -d '{
    "status": "active",
    "reason": "Carte retrouvée"
  }'
```

---

## ✅ Tests

Utilisez le script de test fourni :

```bash
python test_nfc_cards_admin.py
```

---

## 🔒 Sécurité

- ✅ Tous les endpoints nécessitent une authentification admin
- ✅ Les tokens JWT sont vérifiés à chaque requête
- ✅ Les changements de statut sont tracés (qui, quand, pourquoi)
- ✅ Soft delete pour éviter la perte de données
- ✅ Validation des données d'entrée

---

## 📝 Notes Importantes

1. **Synchronisation Mobile** : Les TODO dans le code indiquent où implémenter les notifications WebSocket/Push
2. **Historique** : Tous les changements de statut sont tracés avec l'admin qui a effectué le changement
3. **Soft Delete** : Les cartes ne sont jamais supprimées définitivement, seulement désactivées
4. **Carte Principale** : Un utilisateur ne peut avoir qu'une seule carte principale à la fois

---

## 🚀 Prochaines Étapes

1. Implémenter les notifications WebSocket pour la synchronisation en temps réel
2. Ajouter les endpoints mobile pour récupérer les cartes
3. Implémenter les push notifications
4. Ajouter un système d'audit complet pour tracer tous les changements

