#!/usr/bin/env python3
"""
Script de configuration MySQL pour TIM CASH
Crée la base de données et l'utilisateur MySQL nécessaires
"""

import pymysql
import os
import sys
from dotenv import load_dotenv

# Charger les variables d'environnement
load_dotenv()

def get_mysql_config():
    """Récupérer la configuration MySQL depuis les variables d'environnement"""
    return {
        'host': os.getenv('MYSQL_HOST', 'localhost'),
        'port': int(os.getenv('MYSQL_PORT', 3306)),
        'user': 'root',  # Utilisateur admin pour créer la DB
        'password': input("Mot de passe MySQL root: "),
        'charset': 'utf8mb4'
    }

def create_database_and_user():
    """Créer la base de données et l'utilisateur TIM CASH"""
    
    # Configuration
    db_name = os.getenv('MYSQL_DATABASE', 'timcash_db')
    db_user = os.getenv('MYSQL_USER', 'timcash')
    db_password = os.getenv('MYSQL_PASSWORD', 'timcash123')
    
    print("🔧 Configuration MySQL pour TIM CASH")
    print("=" * 50)
    print(f"Base de données: {db_name}")
    print(f"Utilisateur: {db_user}")
    print("=" * 50)
    
    try:
        # Connexion à MySQL
        config = get_mysql_config()
        print(f"📡 Connexion à MySQL sur {config['host']}:{config['port']}...")
        
        connection = pymysql.connect(**config)
        cursor = connection.cursor()
        
        # Créer la base de données
        print(f"🗄️ Création de la base de données '{db_name}'...")
        cursor.execute(f"CREATE DATABASE IF NOT EXISTS {db_name} CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci")
        
        # Créer l'utilisateur
        print(f"👤 Création de l'utilisateur '{db_user}'...")
        cursor.execute(f"CREATE USER IF NOT EXISTS '{db_user}'@'localhost' IDENTIFIED BY '{db_password}'")
        cursor.execute(f"CREATE USER IF NOT EXISTS '{db_user}'@'%' IDENTIFIED BY '{db_password}'")
        
        # Accorder les privilèges
        print(f"🔑 Attribution des privilèges...")
        cursor.execute(f"GRANT ALL PRIVILEGES ON {db_name}.* TO '{db_user}'@'localhost'")
        cursor.execute(f"GRANT ALL PRIVILEGES ON {db_name}.* TO '{db_user}'@'%'")
        cursor.execute("FLUSH PRIVILEGES")
        
        # Vérifier la connexion avec le nouvel utilisateur
        print(f"✅ Test de connexion avec l'utilisateur '{db_user}'...")
        test_config = {
            'host': config['host'],
            'port': config['port'],
            'user': db_user,
            'password': db_password,
            'database': db_name,
            'charset': 'utf8mb4'
        }
        
        test_connection = pymysql.connect(**test_config)
        test_cursor = test_connection.cursor()
        test_cursor.execute("SELECT 1")
        result = test_cursor.fetchone()
        
        if result[0] == 1:
            print("✅ Configuration MySQL réussie!")
            print(f"🔗 URL de connexion: mysql+pymysql://{db_user}:{db_password}@{config['host']}:{config['port']}/{db_name}")
        
        test_connection.close()
        connection.close()
        
        return True
        
    except pymysql.Error as e:
        print(f"❌ Erreur MySQL: {e}")
        return False
    except Exception as e:
        print(f"❌ Erreur: {e}")
        return False

def verify_mysql_installation():
    """Vérifier que MySQL est installé et accessible"""
    try:
        import pymysql
        print("✅ PyMySQL installé")
        return True
    except ImportError:
        print("❌ PyMySQL non installé. Exécutez: pip install pymysql")
        return False

def main():
    """Fonction principale"""
    print("🚀 Configuration MySQL pour TIM CASH")
    print("=" * 50)
    
    # Vérifier les prérequis
    if not verify_mysql_installation():
        sys.exit(1)
    
    # Créer la base de données et l'utilisateur
    if create_database_and_user():
        print("\n🎉 Configuration terminée avec succès!")
        print("\n📋 Prochaines étapes:")
        print("1. Copiez .env.example vers .env")
        print("2. Vérifiez la configuration dans .env")
        print("3. Exécutez: python recreate_database.py")
        print("4. Démarrez le serveur: python -m uvicorn main:app --reload --port 8001")
    else:
        print("\n❌ Échec de la configuration")
        sys.exit(1)

if __name__ == "__main__":
    main()
