#!/usr/bin/env python3
"""
Script pour peupler la base de données TIM CASH avec des données d'exemple
"""
import sys
import os
sys.path.append('.')

from database import SessionLocal
from models.models import *
import uuid
from datetime import datetime, timedelta
import json

def seed_countries_and_cities():
    """Créer les pays et villes TIM CASH"""
    db = SessionLocal()
    
    try:
        # Vérifier si des pays existent déjà
        if db.query(Country).count() > 0:
            print("✅ Pays et villes déjà présents")
            return
        
        print("🌍 Création des pays et villes...")
        
        # Côte d'Ivoire
        ci = Country(
            id=str(uuid.uuid4()),
            name="Côte d'Ivoire",
            code="CI",
            currency="FCFA",
            currency_symbol="FCFA",
            flag="🇨🇮",
            total_users=2847,
            total_volume=562000000.0
        )
        db.add(ci)
        db.flush()
        
        # Villes de Côte d'Ivoire
        cities_ci = [
            {"name": "Abidjan", "branch_code": "TIM-CI-ABJ", "users": 1847, "volume": 450000000.0},
            {"name": "Yamoussoukro", "branch_code": "TIM-CI-YAM", "users": 500, "volume": 62000000.0},
            {"name": "Bouaké", "branch_code": "TIM-CI-BOU", "users": 500, "volume": 50000000.0}
        ]
        
        for city_data in cities_ci:
            city = City(
                id=str(uuid.uuid4()),
                country_id=ci.id,
                **city_data
            )
            db.add(city)
        
        # Sénégal
        sn = Country(
            id=str(uuid.uuid4()),
            name="Sénégal",
            code="SN",
            currency="FCFA",
            currency_symbol="FCFA",
            flag="🇸🇳",
            total_users=1250,
            total_volume=185000000.0
        )
        db.add(sn)
        db.flush()
        
        # Villes du Sénégal
        cities_sn = [
            {"name": "Dakar", "branch_code": "TIM-SN-DKR", "users": 950, "volume": 140000000.0},
            {"name": "Thiès", "branch_code": "TIM-SN-THI", "users": 300, "volume": 45000000.0}
        ]
        
        for city_data in cities_sn:
            city = City(
                id=str(uuid.uuid4()),
                country_id=sn.id,
                **city_data
            )
            db.add(city)
        
        db.commit()
        print("✅ Pays et villes créés avec succès")
        
    except Exception as e:
        print(f"❌ Erreur lors de la création des pays: {e}")
        db.rollback()
    finally:
        db.close()

def seed_commission_rates():
    """Créer les taux de commission par défaut"""
    db = SessionLocal()
    
    try:
        # Vérifier si des taux existent déjà
        if db.query(CommissionRate).count() > 0:
            print("✅ Taux de commission déjà présents")
            return
        
        print("💰 Création des taux de commission...")
        
        commission_rates = [
            {
                "id": str(uuid.uuid4()),
                "name": "Commission CinetPay",
                "rate_type": "percentage",
                "value": 0.5,
                "category": "recharge_cinetpay",
                "description": "Commission sur les recharges CinetPay",
                "updated_by": "system"
            },
            {
                "id": str(uuid.uuid4()),
                "name": "Frais de retrait",
                "rate_type": "percentage", 
                "value": 2.0,
                "category": "withdrawal",
                "description": "Frais sur les retraits",
                "updated_by": "system"
            },
            {
                "id": str(uuid.uuid4()),
                "name": "Commission TIM BUSINESS",
                "rate_type": "percentage",
                "value": 2.0,
                "category": "recharge_by_business",
                "description": "Commission pour recharges effectuées par TIM BUSINESS",
                "updated_by": "system"
            },
            {
                "id": str(uuid.uuid4()),
                "name": "Taux d'intérêt prêt",
                "rate_type": "percentage",
                "value": 6.0,
                "category": "loan_interest",
                "description": "Taux d'intérêt sur les prêts SOS",
                "updated_by": "system"
            },
            {
                "id": str(uuid.uuid4()),
                "name": "Pénalité de retard",
                "rate_type": "percentage",
                "value": 15.0,
                "category": "loan_penalty",
                "description": "Pénalité en cas de retard de remboursement",
                "updated_by": "system"
            },
            {
                "id": str(uuid.uuid4()),
                "name": "Frais virement bancaire",
                "rate_type": "fixed",
                "value": 500.0,
                "category": "bank_transfer",
                "description": "Frais pour virements vers banques",
                "updated_by": "system"
            }
        ]
        
        for rate_data in commission_rates:
            rate = CommissionRate(**rate_data)
            db.add(rate)
        
        db.commit()
        print("✅ Taux de commission créés avec succès")
        
    except Exception as e:
        print(f"❌ Erreur lors de la création des taux: {e}")
        db.rollback()
    finally:
        db.close()

def seed_device_capabilities():
    """Créer les capacités d'appareils pour analytics NFC"""
    db = SessionLocal()
    
    try:
        # Vérifier si des appareils existent déjà
        if db.query(DeviceCapability).count() > 0:
            print("✅ Capacités d'appareils déjà présentes")
            return
        
        print("📱 Création des capacités d'appareils...")
        
        devices = [
            {
                "id": str(uuid.uuid4()),
                "device_model": "Samsung Galaxy A54",
                "brand": "Samsung",
                "has_nfc": True,
                "nfc_enabled": True,
                "os_version": "Android 13",
                "user_count": 1247,
                "transaction_count": 15680,
                "success_rate": 98.7,
                "location": "Abidjan Centre",
                "country": "Côte d'Ivoire",
                "city": "Abidjan"
            },
            {
                "id": str(uuid.uuid4()),
                "device_model": "iPhone 14",
                "brand": "Apple",
                "has_nfc": True,
                "nfc_enabled": False,
                "os_version": "iOS 17.2",
                "user_count": 892,
                "transaction_count": 8920,
                "success_rate": 95.2,
                "location": "Cocody",
                "country": "Côte d'Ivoire", 
                "city": "Abidjan"
            },
            {
                "id": str(uuid.uuid4()),
                "device_model": "Tecno Spark 10",
                "brand": "Tecno",
                "has_nfc": False,
                "nfc_enabled": False,
                "os_version": "Android 12",
                "user_count": 2156,
                "transaction_count": 12450,
                "success_rate": 89.4,
                "location": "Yopougon",
                "country": "Côte d'Ivoire",
                "city": "Abidjan"
            },
            {
                "id": str(uuid.uuid4()),
                "device_model": "Xiaomi Redmi Note 12",
                "brand": "Xiaomi",
                "has_nfc": True,
                "nfc_enabled": True,
                "os_version": "Android 13",
                "user_count": 756,
                "transaction_count": 9870,
                "success_rate": 97.1,
                "location": "Adjamé",
                "country": "Côte d'Ivoire",
                "city": "Abidjan"
            },
            {
                "id": str(uuid.uuid4()),
                "device_model": "Infinix Note 30",
                "brand": "Infinix",
                "has_nfc": False,
                "nfc_enabled": False,
                "os_version": "Android 12",
                "user_count": 1834,
                "transaction_count": 11250,
                "success_rate": 91.8,
                "location": "Marcory",
                "country": "Côte d'Ivoire",
                "city": "Abidjan"
            }
        ]
        
        for device_data in devices:
            device = DeviceCapability(**device_data)
            db.add(device)
        
        db.commit()
        print("✅ Capacités d'appareils créées avec succès")
        
    except Exception as e:
        print(f"❌ Erreur lors de la création des appareils: {e}")
        db.rollback()
    finally:
        db.close()

def seed_sample_users():
    """Créer des utilisateurs d'exemple"""
    db = SessionLocal()
    
    try:
        # Vérifier si des utilisateurs existent déjà (en plus de l'admin)
        if db.query(User).count() > 1:
            print("✅ Utilisateurs d'exemple déjà présents")
            return
        
        print("👥 Création d'utilisateurs d'exemple...")
        
        import bcrypt
        
        users_data = [
            {
                "username": "marie.kouassi",
                "email": "marie.kouassi@timcash.ci",
                "full_name": "Marie Kouassi",
                "phone": "+225 07 12 34 56 78",
                "tim_account_type": TimAccountType.TIM_MAXI,
                "balance": 2847000.0  # 2.847M FCFA
            },
            {
                "username": "jean.baptiste",
                "email": "jean.baptiste@timcash.ci", 
                "full_name": "Jean Baptiste Traoré",
                "phone": "+225 05 98 76 54 32",
                "tim_account_type": TimAccountType.TIM_MINI,
                "balance": 125000.0  # 125K FCFA
            },
            {
                "username": "fatou.diallo",
                "email": "fatou.diallo@timcash.ci",
                "full_name": "Fatou Diallo",
                "phone": "+221 77 123 45 67",
                "tim_account_type": TimAccountType.TIM_BUSINESS,
                "balance": 5500000.0  # 5.5M FCFA
            }
        ]
        
        for user_data in users_data:
            # Hasher le mot de passe par défaut
            hashed_password = bcrypt.hashpw("password123".encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
            
            user = User(
                id=str(uuid.uuid4()),
                username=user_data["username"],
                email=user_data["email"],
                full_name=user_data["full_name"],
                phone=user_data["phone"],
                hashed_password=hashed_password,
                tim_account_type=user_data["tim_account_type"]
            )
            
            db.add(user)
            db.flush()
            
            # Créer le wallet
            wallet = Wallet(
                id=str(uuid.uuid4()),
                user_id=user.id,
                balance=user_data["balance"],
                max_balance=get_account_limit(user_data["tim_account_type"])
            )
            
            db.add(wallet)
        
        db.commit()
        print("✅ Utilisateurs d'exemple créés avec succès")
        print("   📧 Emails: marie.kouassi@timcash.ci, jean.baptiste@timcash.ci, fatou.diallo@timcash.ci")
        print("   🔑 Mot de passe: password123")
        
    except Exception as e:
        print(f"❌ Erreur lors de la création des utilisateurs: {e}")
        db.rollback()
    finally:
        db.close()

def get_account_limit(account_type: TimAccountType):
    """Obtenir la limite selon le type de compte"""
    limits = {
        TimAccountType.TIM_MINI: 500000.0,      # 500K FCFA
        TimAccountType.TIM_MAXI: 12000000.0,    # 12M FCFA
        TimAccountType.TIM_BUSINESS: None       # Illimité
    }
    return limits.get(account_type)

def seed_sample_transactions():
    """Créer des transactions d'exemple"""
    db = SessionLocal()
    
    try:
        # Vérifier si des transactions existent déjà
        if db.query(Transaction).count() > 0:
            print("✅ Transactions d'exemple déjà présentes")
            return
        
        print("💳 Création de transactions d'exemple...")
        
        # Récupérer les utilisateurs
        users = db.query(User).filter(User.email != "admin@timcash.ci").all()
        
        if not users:
            print("⚠️ Aucun utilisateur trouvé pour créer des transactions")
            return
        
        transactions_data = [
            {
                "user_id": users[0].id,  # Marie Kouassi
                "transaction_type": TransactionType.RECHARGE_CINETPAY,
                "amount": 100000.0,
                "commission_amount": 500.0,
                "commission_rate": 0.5,
                "status": TransactionStatus.COMPLETED,
                "description": "Recharge CinetPay - Orange Money",
                "reference": f"REF-{uuid.uuid4().hex[:8].upper()}",
                "payment_method": PaymentMethod.MOBILE_MONEY,
                "completed_at": datetime.utcnow() - timedelta(hours=2)
            },
            {
                "user_id": users[1].id,  # Jean Baptiste
                "transaction_type": TransactionType.NFC_DEPOSIT,
                "amount": 25000.0,
                "status": TransactionStatus.COMPLETED,
                "description": "Paiement NFC - Boutique Centrale",
                "reference": f"NFC-{uuid.uuid4().hex[:8].upper()}",
                "merchant_name": "Boutique Centrale",
                "location": "Abidjan Centre",
                "completed_at": datetime.utcnow() - timedelta(minutes=30)
            },
            {
                "user_id": users[0].id,  # Marie Kouassi
                "transaction_type": TransactionType.PRET_MOBILE,
                "amount": 8500.0,
                "status": TransactionStatus.COMPLETED,
                "description": "Prêt SOS - Urgence médicale",
                "reference": f"SOS-{uuid.uuid4().hex[:8].upper()}",
                "completed_at": datetime.utcnow() - timedelta(days=1)
            }
        ]
        
        for trans_data in transactions_data:
            transaction = Transaction(
                id=str(uuid.uuid4()),
                **trans_data
            )
            db.add(transaction)
        
        db.commit()
        print("✅ Transactions d'exemple créées avec succès")
        
    except Exception as e:
        print(f"❌ Erreur lors de la création des transactions: {e}")
        db.rollback()
    finally:
        db.close()

def main():
    """Fonction principale pour peupler la base de données"""
    print("🌱 Peuplement de la base de données TIM CASH")
    print("=" * 50)
    
    steps = [
        ("Pays et villes", seed_countries_and_cities),
        ("Taux de commission", seed_commission_rates),
        ("Capacités d'appareils", seed_device_capabilities),
        ("Utilisateurs d'exemple", seed_sample_users),
        ("Transactions d'exemple", seed_sample_transactions)
    ]
    
    for step_name, step_func in steps:
        print(f"\n🔄 {step_name}...")
        try:
            step_func()
        except Exception as e:
            print(f"❌ Erreur dans {step_name}: {e}")
    
    print("\n" + "=" * 50)
    print("🎉 Peuplement terminé !")
    print("\n📋 Données créées:")
    print("   🌍 Pays: Côte d'Ivoire, Sénégal")
    print("   🏙️ Villes: Abidjan, Yamoussoukro, Bouaké, Dakar, Thiès")
    print("   💰 Taux de commission: CinetPay, TIM BUSINESS, Prêts")
    print("   📱 Appareils: Samsung, iPhone, Tecno, Xiaomi, Infinix")
    print("   👥 Utilisateurs: marie.kouassi, jean.baptiste, fatou.diallo")
    print("   💳 Transactions: Recharges, NFC, Prêts SOS")

if __name__ == "__main__":
    main()
