"""
Schémas pour la gestion des cartes NFC
"""
from pydantic import BaseModel, Field
from typing import Optional, List
from datetime import datetime

class NFCCardBase(BaseModel):
    """Schéma de base pour une carte NFC"""
    card_uid: str = Field(..., description="UID unique de la carte NFC")
    card_number: str = Field(..., description="Numéro de carte lisible")
    card_type: str = Field(default="physical", description="Type de carte (physical, virtual)")
    status: str = Field(default="active", description="Statut de la carte")
    max_balance: float = Field(default=50000.0, description="Limite de solde sur la carte")
    is_primary: bool = Field(default=False, description="Carte principale de l'utilisateur")

class NFCCardCreate(NFCCardBase):
    """Schéma pour créer une carte NFC"""
    user_id: str = Field(..., description="ID de l'utilisateur propriétaire")

class NFCCardResponse(NFCCardBase):
    """Schéma de réponse pour une carte NFC"""
    id: str
    user_id: str
    balance: float
    last_used_at: Optional[datetime] = None
    created_at: datetime
    updated_at: Optional[datetime] = None
    
    class Config:
        from_attributes = True

class NFCCardLinkRequest(BaseModel):
    """Schéma pour lier une carte NFC à un utilisateur"""
    card_uid: str = Field(..., description="UID de la carte à lier")
    card_number: str = Field(..., description="Numéro de la carte")
    user_phone: str = Field(..., description="Téléphone de l'utilisateur")

class NFCCardUnlinkRequest(BaseModel):
    """Schéma pour délier une carte NFC"""
    card_uid: str = Field(..., description="UID de la carte à délier")
    reason: Optional[str] = Field(None, description="Raison du déliage")

class NFCCardUpdateRequest(BaseModel):
    """Schéma pour mettre à jour une carte NFC"""
    status: Optional[str] = Field(None, description="Nouveau statut")
    max_balance: Optional[float] = Field(None, description="Nouvelle limite de solde")
    is_primary: Optional[bool] = Field(None, description="Définir comme carte principale")

class NFCCardTransactionRequest(BaseModel):
    """Schéma pour une transaction sur carte NFC"""
    card_uid: str = Field(..., description="UID de la carte")
    amount: float = Field(..., description="Montant de la transaction")
    transaction_type: str = Field(..., description="Type de transaction (deposit, withdrawal)")
    merchant_name: Optional[str] = Field(None, description="Nom du commerçant")
    location: Optional[str] = Field(None, description="Localisation")

class NFCCardReadRequest(BaseModel):
    """Schéma pour lire les données d'une carte NFC"""
    card_uid: str = Field(..., description="UID de la carte à lire")
    device_info: Optional[str] = Field(None, description="Informations de l'appareil")

class NFCCardWriteRequest(BaseModel):
    """Schéma pour écrire sur une carte NFC"""
    card_uid: str = Field(..., description="UID de la carte")
    card_data: dict = Field(..., description="Données à écrire sur la carte")
    device_info: Optional[str] = Field(None, description="Informations de l'appareil")

class NFCCardTransactionResponse(BaseModel):
    """Schéma de réponse pour une transaction carte NFC"""
    id: str
    card_id: str
    transaction_id: str
    operation_type: str
    card_data: Optional[str] = None
    device_info: Optional[str] = None
    location: Optional[str] = None
    created_at: datetime
    
    class Config:
        from_attributes = True

class NFCCardBalanceResponse(BaseModel):
    """Schéma de réponse pour le solde d'une carte"""
    card_uid: str
    card_number: str
    balance: float
    max_balance: float
    user_name: str
    user_phone: str
    last_used_at: Optional[datetime] = None

class NFCCardListResponse(BaseModel):
    """Schéma de réponse pour la liste des cartes d'un utilisateur"""
    cards: List[NFCCardResponse]
    total_cards: int
    primary_card: Optional[NFCCardResponse] = None
