"""
Schémas pour la gestion des commissions
"""
from pydantic import BaseModel, Field
from typing import Optional
from datetime import datetime

class CommissionRateResponse(BaseModel):
    """Réponse pour un taux de commission"""
    id: str
    name: str
    category: str
    value: float = Field(..., description="Taux de commission (0.0 à 1.0)")
    description: Optional[str] = None
    is_active: bool = True
    updated_by: str
    created_at: datetime
    updated_at: Optional[datetime] = None

class UpdateCommissionRequest(BaseModel):
    """Requête pour mettre à jour un taux de commission"""
    value: float = Field(..., ge=0, le=1, description="Nouveau taux de commission (0.0 à 1.0)")
    description: Optional[str] = Field(None, description="Description du taux de commission")

class UpdateCommissionByIdRequest(BaseModel):
    """Requête pour mettre à jour une commission par ID"""
    name: Optional[str] = Field(None, description="Nom de la commission")
    rate_type: Optional[str] = Field(None, description="Type de taux (percentage, fixed)")
    value: float = Field(..., ge=0, description="Valeur du taux")
    category: Optional[str] = Field(None, description="Catégorie de la commission")
    description: Optional[str] = Field(None, description="Description")
    min_amount: Optional[float] = Field(None, description="Montant minimum")
    max_amount: Optional[float] = Field(None, description="Montant maximum")
    is_active: Optional[bool] = Field(None, description="Statut actif/inactif")

class CommissionCalculationResponse(BaseModel):
    """Réponse pour le calcul de commission"""
    amount: float
    category: str
    total_commission: float
    cinetpay_commission: Optional[float] = None
    timcash_commission: Optional[float] = None
    net_amount: float
    total_payment: float

class RechargeCommissionRates(BaseModel):
    """Taux de commission pour les recharges"""
    total_rate: float = Field(..., description="Taux total (ex: 0.03 pour 3%)")
    cinetpay_rate: float = Field(..., description="Taux CinetPay (ex: 0.01 pour 1%)")
    timcash_rate: float = Field(..., description="Taux TimCash (ex: 0.02 pour 2%)")

class RechargeCommissionBreakdown(BaseModel):
    """Détail de la répartition des commissions de recharge"""
    total_amount: float = Field(..., description="Montant total de la transaction")
    total_commission: float = Field(..., description="Commission totale (3%)")
    cinetpay_commission: float = Field(..., description="Part CinetPay (1%)")
    timcash_commission: float = Field(..., description="Part TimCash (2%)")
    net_amount: float = Field(..., description="Montant net après commissions")
    
class RechargeCommissionResponse(BaseModel):
    """Réponse avec détail des commissions de recharge"""
    success: bool
    message: str
    breakdown: RechargeCommissionBreakdown
    transaction_id: Optional[str] = None
