# API des Cartes Virtuelles TIM CASH

## 📋 Vue d'ensemble

L'API des cartes virtuelles permet aux utilisateurs de l'application mobile TIM CASH (TIM MINI, TIM MAXI, TIM BUSINESS) d'acheter des cartes virtuelles personnalisées avec différentes couleurs.

## 🎨 Couleurs Disponibles

**4 couleurs actives à 650 FCFA chacune :**

| Couleur | Code | Prix | Type |
|---------|------|------|------|
| **GOLD** | `gold` | 650 FCFA | Or classique |
| **AMBER** | `amber` | 650 FCFA | Ambre (vert-jaune dégradé) |
| **GREEN** | `green` | 650 FCFA | Vert pur |
| **INFINITE** | `infinite` | 650 FCFA | Noir premium ⭐ |

**6 couleurs désactivées (pour compatibilité future) :**
- SILVER, BLACK, BLUE, RED, PURPLE, ORANGE

## 🔄 Workflow d'Achat

```
1. Utilisateur sélectionne une couleur dans l'app mobile
   ↓
2. API: GET /api/virtual-cards/colors
   → Récupère les couleurs disponibles et leurs prix
   ↓
3. API: POST /api/virtual-cards/purchase
   → Initie l'achat et crée une transaction
   → Génère une URL de paiement CinetPay
   ↓
4. Utilisateur redirigé vers CinetPay
   → Paiement par Mobile Money (Wave, Orange, MTN, Moov)
   ↓
5. CinetPay Callback: POST /api/virtual-cards/callback
   → Vérifie le paiement
   → Crée automatiquement la carte virtuelle avec la couleur choisie
   ↓
6. Utilisateur redirigé: GET /api/virtual-cards/return
   → Retour vers l'application mobile
   ↓
7. Carte virtuelle créée et affichée dans l'app
```

## 📡 Endpoints API

### 1. GET `/api/virtual-cards/colors`

Récupère les couleurs de cartes virtuelles disponibles avec leurs prix.

**Authentification:** Requise (JWT Token)

**Accessible par:** TIM MINI, TIM MAXI, TIM BUSINESS

**Réponse:**
```json
{
  "colors": [
    {
      "card_color": "gold",
      "base_price": 650.0,
      "price_for_user": 650.0,
      "commission_amount": 0.0,
      "total_amount": 650.0,
      "is_available": true,
      "is_premium": false
    },
    {
      "card_color": "infinite",
      "base_price": 650.0,
      "price_for_user": 650.0,
      "commission_amount": 0.0,
      "total_amount": 650.0,
      "is_available": true,
      "is_premium": true
    }
  ],
  "total_colors": 4
}
```

### 2. POST `/api/virtual-cards/purchase`

Initie l'achat d'une carte virtuelle avec paiement CinetPay.

**Authentification:** Requise (JWT Token)

**Accessible par:** TIM MINI, TIM MAXI, TIM BUSINESS

**Requête:**
```json
{
  "card_color": "gold",
  "device_info": "{\"model\": \"iPhone 12\", \"os\": \"iOS 15\"}"
}
```

**Réponse (succès):**
```json
{
  "success": true,
  "message": "Achat initié avec succès",
  "purchase_id": "uuid-xxx-xxx",
  "payment_url": "https://checkout.cinetpay.com/payment/xxx",
  "payment_token": "token-xxx",
  "amount": 650.0,
  "total_amount": 650.0,
  "status": "payment_initiated"
}
```

**Réponse (échec):**
```json
{
  "success": false,
  "message": "Échec initiation paiement",
  "purchase_id": "uuid-xxx-xxx",
  "status": "failed"
}
```

### 3. GET `/api/virtual-cards/history`

Récupère l'historique des achats de cartes virtuelles de l'utilisateur.

**Authentification:** Requise (JWT Token)

**Réponse:**
```json
{
  "purchases": [
    {
      "purchase_id": "uuid-xxx",
      "card_color": "gold",
      "amount": 650.0,
      "total_amount": 650.0,
      "status": "card_created",
      "created_at": "2025-11-06T14:00:00Z",
      "card_id": "card-uuid-xxx"
    }
  ],
  "total_count": 5,
  "total_spent": 3250.0
}
```

### 4. GET `/api/virtual-cards/purchase/{purchase_id}`

Vérifie le statut d'un achat spécifique.

**Authentification:** Requise (JWT Token)

**Réponse:**
```json
{
  "purchase_id": "uuid-xxx",
  "status": "card_created",
  "card_id": "card-uuid-xxx",
  "payment_completed": true,
  "card_created": true
}
```

### 5. POST `/api/virtual-cards/callback`

Webhook appelé par CinetPay après le paiement.

**Authentification:** Non requise (appelé par CinetPay)

**Requête (de CinetPay):**
```json
{
  "transaction_id": "VCARD-xxx",
  "status": "ACCEPTED",
  "amount": 650,
  "payment_method": "WAVE",
  "operator_id": "xxx"
}
```

**Actions:**
- Vérifie le statut du paiement
- Met à jour le statut de l'achat
- Crée automatiquement la carte virtuelle avec la couleur choisie
- Enregistre la couleur dans `nfc_cards.card_color`

### 6. GET `/api/virtual-cards/return`

Page de retour après paiement CinetPay.

**Paramètres:**
- `transaction_id`: ID de la transaction
- `status`: Statut du paiement (success/failed)

**Redirection:** Vers l'application mobile avec le statut

## 🗄️ Structure de la Base de Données

### Table `virtual_card_prices`

Configuration des prix des cartes virtuelles.

```sql
CREATE TABLE virtual_card_prices (
    id VARCHAR(50) PRIMARY KEY,
    card_color ENUM('gold', 'amber', 'green', 'infinite', ...) NOT NULL UNIQUE,
    base_price FLOAT NOT NULL DEFAULT 650.0,
    tim_mini_price FLOAT NULL,
    tim_maxi_price FLOAT NULL,
    tim_business_price FLOAT NULL,
    commission_rate FLOAT DEFAULT 0.0,
    is_available BOOLEAN DEFAULT TRUE,
    is_premium BOOLEAN DEFAULT FALSE,
    max_purchases_per_user INT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME ON UPDATE CURRENT_TIMESTAMP
);
```

### Table `virtual_card_purchases`

Historique des achats de cartes virtuelles.

```sql
CREATE TABLE virtual_card_purchases (
    id VARCHAR(50) PRIMARY KEY,
    user_id VARCHAR(50) NOT NULL,
    card_color ENUM('gold', 'amber', 'green', 'infinite', ...) NOT NULL,
    card_id VARCHAR(50) NULL,
    amount FLOAT NOT NULL,
    commission_amount FLOAT DEFAULT 0.0,
    total_amount FLOAT NOT NULL,
    status ENUM('pending', 'payment_initiated', 'payment_completed', 'card_created', 'failed', 'cancelled'),
    payment_transaction_id VARCHAR(100) UNIQUE,
    cinetpay_transaction_id VARCHAR(100),
    cinetpay_payment_url TEXT,
    cinetpay_payment_token VARCHAR(255),
    payment_method VARCHAR(50),
    payment_phone VARCHAR(20),
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    payment_initiated_at DATETIME,
    payment_completed_at DATETIME,
    card_created_at DATETIME,
    failed_at DATETIME,
    failure_reason TEXT,
    device_info TEXT,
    ip_address VARCHAR(50),
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (card_id) REFERENCES nfc_cards(id)
);
```

### Modification Table `nfc_cards`

Ajout de la colonne `card_color` pour stocker la couleur de la carte virtuelle.

```sql
ALTER TABLE nfc_cards 
ADD COLUMN card_color ENUM('gold', 'amber', 'green', 'infinite', ...) NULL;

CREATE INDEX idx_card_color ON nfc_cards(card_color);
```

## 🔧 Configuration

### Variables d'Environnement

```env
# CinetPay Configuration
CINETPAY_API_KEY=your-api-key
CINETPAY_SECRET_KEY=your-secret-key
CINETPAY_SITE_ID=your-site-id
BACKEND_URL=http://localhost:8001
```

## ✅ Tests

Exécuter le script de test :

```bash
cd frontnew/backend
python test_virtual_cards_api.py
```

**Résultats attendus :**
- ✅ 4 couleurs disponibles à 650 FCFA
- ✅ Initiation d'achat (échoue si CinetPay non configuré)
- ✅ Historique des achats
- ✅ Statut des achats

## 📱 Intégration Mobile

Dans l'app mobile Flutter (`timcash_app`), le bouton "Obtenir une carte virtuelle" se trouve dans :

**Fichier:** `timcash_app/lib/presentation/wallets/wallet_screen.dart`

```dart
final userCards = [
  {'type': 'GOLD', 'price': 650},      // Or classique
  {'type': 'AMBER', 'price': 650},     // Ambre
  {'type': 'GREEN', 'price': 650},     // Vert pur
  {'type': 'INFINITE', 'price': 650},  // Noir premium
];
```

## 🎯 Prochaines Étapes

1. ✅ Backend API créé et testé
2. ⏳ Configurer les credentials CinetPay en production
3. ⏳ Intégrer l'API dans l'application mobile Flutter
4. ⏳ Tester le flux complet de paiement
5. ⏳ Afficher la carte virtuelle avec la couleur choisie dans l'app

## 📝 Notes

- **Prix unique:** Toutes les couleurs actives coûtent 650 FCFA
- **Pas de commission:** `commission_rate = 0.0` pour l'achat de cartes
- **Carte premium:** Seule la couleur INFINITE est marquée comme premium
- **Historique complet:** Toutes les transactions sont enregistrées avec timestamps
- **Couleur sauvegardée:** La couleur choisie est stockée dans `nfc_cards.card_color`

