#!/usr/bin/env python3
"""
Configuration MySQL pour TIM CASH avec WAMP
WAMP utilise généralement root sans mot de passe par défaut
"""

import pymysql
import sys
import os
from dotenv import load_dotenv, set_key

load_dotenv()

def setup_mysql_wamp():
    """Configuration MySQL pour WAMP"""
    print("\n" + "=" * 60)
    print("🔧 Configuration MySQL TIM CASH (WAMP)")
    print("=" * 60 + "\n")
    
    # Configuration WAMP par défaut
    print("ℹ️  WAMP utilise généralement:")
    print("   - Utilisateur: root")
    print("   - Mot de passe: (vide) ou root")
    print()
    
    # Essayer avec mot de passe vide
    root_password = input("Mot de passe root MySQL [vide par défaut, appuyez sur Entrée]: ").strip()
    
    # Configuration
    db_name = 'timcash_db'
    db_user = 'timcash'
    db_password = 'timcash123'
    
    print(f"\n📊 Configuration:")
    print(f"   Base de données: {db_name}")
    print(f"   Utilisateur: {db_user}")
    print(f"   Mot de passe: {db_password}")
    
    try:
        # Connexion root
        print("\n🔌 Connexion à MySQL...")
        connection = pymysql.connect(
            host='localhost',
            port=3306,
            user='root',
            password=root_password,
            charset='utf8mb4'
        )
        cursor = connection.cursor()
        print("✅ Connexion réussie!")
        
        # Créer la base de données
        print(f"\n🗄️  Création de la base '{db_name}'...")
        cursor.execute(f"DROP DATABASE IF EXISTS {db_name}")
        cursor.execute(f"CREATE DATABASE {db_name} CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci")
        print(f"✅ Base '{db_name}' créée")
        
        # Créer l'utilisateur
        print(f"\n👤 Configuration de l'utilisateur '{db_user}'...")
        try:
            cursor.execute(f"DROP USER IF EXISTS '{db_user}'@'localhost'")
            cursor.execute(f"DROP USER IF EXISTS '{db_user}'@'%'")
        except:
            pass
        
        cursor.execute(f"CREATE USER '{db_user}'@'localhost' IDENTIFIED BY '{db_password}'")
        cursor.execute(f"CREATE USER '{db_user}'@'%' IDENTIFIED BY '{db_password}'")
        print(f"✅ Utilisateur '{db_user}' créé")
        
        # Privilèges
        print(f"\n🔑 Attribution des privilèges...")
        cursor.execute(f"GRANT ALL PRIVILEGES ON {db_name}.* TO '{db_user}'@'localhost'")
        cursor.execute(f"GRANT ALL PRIVILEGES ON {db_name}.* TO '{db_user}'@'%'")
        cursor.execute("FLUSH PRIVILEGES")
        print("✅ Privilèges accordés")
        
        connection.close()
        
        # Test de connexion
        print(f"\n🔍 Test de connexion avec '{db_user}'...")
        test_conn = pymysql.connect(
            host='localhost',
            port=3306,
            user=db_user,
            password=db_password,
            database=db_name,
            charset='utf8mb4'
        )
        test_cursor = test_conn.cursor()
        test_cursor.execute("SELECT 1")
        result = test_cursor.fetchone()
        
        if result[0] == 1:
            print("✅ Test réussi!")
        
        test_conn.close()
        
        # Mettre à jour .env
        print("\n📝 Mise à jour du fichier .env...")
        
        env_file = '.env'
        if not os.path.exists(env_file):
            print("   Création du fichier .env depuis .env.mysql...")
            with open('.env.mysql', 'r', encoding='utf-8') as source:
                with open('.env', 'w', encoding='utf-8') as dest:
                    dest.write(source.read())
        
        # URL de connexion
        database_url = f"mysql+pymysql://{db_user}:{db_password}@localhost:3306/{db_name}"
        
        # Mettre à jour
        set_key(env_file, 'DATABASE_URL', database_url)
        set_key(env_file, 'MYSQL_HOST', 'localhost')
        set_key(env_file, 'MYSQL_PORT', '3306')
        set_key(env_file, 'MYSQL_USER', db_user)
        set_key(env_file, 'MYSQL_PASSWORD', db_password)
        set_key(env_file, 'MYSQL_DATABASE', db_name)
        
        print(f"✅ Fichier .env mis à jour")
        print(f"\n🔗 URL de connexion:")
        print(f"   {database_url}")
        
        print("\n" + "=" * 60)
        print("🎉 Configuration MySQL terminée avec succès!")
        print("=" * 60)
        print("\n📋 Prochaine étape:")
        print("   python migrate_sqlite_to_mysql.py")
        print("=" * 60 + "\n")
        
        return True
        
    except pymysql.Error as e:
        print(f"\n❌ Erreur MySQL: {e}")
        print("\n💡 Vérifiez:")
        print("   1. WAMP est démarré (icône verte)")
        print("   2. MySQL tourne (port 3306)")
        print("   3. Mot de passe root correct")
        return False
    except Exception as e:
        print(f"\n❌ Erreur: {e}")
        return False

if __name__ == "__main__":
    try:
        if setup_mysql_wamp():
            sys.exit(0)
        else:
            sys.exit(1)
    except KeyboardInterrupt:
        print("\n\n⚠️ Configuration annulée")
        sys.exit(0)
