#!/usr/bin/env python3
"""
Configuration de l'utilisateur MySQL pour TIM CASH
"""

import pymysql
import getpass
import sys

def setup_mysql_user():
    """Configurer l'utilisateur MySQL pour TIM CASH"""
    print("🛠️ Configuration de l'utilisateur MySQL pour TIM CASH")
    print("=" * 60)
    
    # Demander les informations de connexion root
    print("📝 Veuillez entrer les informations de connexion MySQL root:")
    root_password = getpass.getpass("Mot de passe root MySQL: ")
    
    try:
        # Connexion en tant que root
        print("🔌 Connexion à MySQL en tant que root...")
        connection = pymysql.connect(
            host='localhost',
            user='root',
            password=root_password,
            charset='utf8mb4'
        )
        
        print("✅ Connexion root réussie")
        
        with connection.cursor() as cursor:
            # Créer l'utilisateur timcash
            print("👤 Création de l'utilisateur 'timcash'...")
            try:
                cursor.execute("DROP USER IF EXISTS 'timcash'@'localhost'")
                cursor.execute("CREATE USER 'timcash'@'localhost' IDENTIFIED BY 'timcash123'")
                print("✅ Utilisateur 'timcash' créé")
            except Exception as e:
                print(f"⚠️ Utilisateur existe déjà ou erreur: {e}")
            
            # Accorder tous les privilèges
            print("🔑 Attribution des privilèges...")
            cursor.execute("GRANT ALL PRIVILEGES ON *.* TO 'timcash'@'localhost'")
            cursor.execute("FLUSH PRIVILEGES")
            print("✅ Privilèges accordés")
            
            # Créer la base de données
            print("🗄️ Création de la base de données 'timcash_db'...")
            cursor.execute("CREATE DATABASE IF NOT EXISTS timcash_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci")
            print("✅ Base de données 'timcash_db' créée")
            
            # Vérifier la configuration
            print("🔍 Vérification de la configuration...")
            cursor.execute("SELECT User, Host FROM mysql.user WHERE User = 'timcash'")
            user_result = cursor.fetchone()
            
            cursor.execute("SHOW DATABASES LIKE 'timcash_db'")
            db_result = cursor.fetchone()
            
            if user_result and db_result:
                print("✅ Configuration MySQL complète !")
                print(f"   👤 Utilisateur: {user_result[0]}@{user_result[1]}")
                print(f"   🗄️ Base de données: {db_result[0]}")
                return True
            else:
                print("❌ Problème de configuration")
                return False
        
    except pymysql.Error as e:
        print(f"❌ Erreur MySQL: {e}")
        print("\n💡 Vérifiez que:")
        print("   1. MySQL est démarré")
        print("   2. Le mot de passe root est correct")
        print("   3. Vous avez les droits d'administration")
        return False
    
    except Exception as e:
        print(f"❌ Erreur générale: {e}")
        return False
    
    finally:
        if 'connection' in locals():
            connection.close()

def test_timcash_connection():
    """Tester la connexion avec l'utilisateur timcash"""
    print("\n🧪 Test de connexion avec l'utilisateur 'timcash'...")
    
    try:
        connection = pymysql.connect(
            host='localhost',
            user='timcash',
            password='timcash123',
            database='timcash_db',
            charset='utf8mb4'
        )
        
        with connection.cursor() as cursor:
            cursor.execute("SELECT DATABASE()")
            result = cursor.fetchone()
            print(f"✅ Connexion réussie à la base: {result[0]}")
        
        connection.close()
        return True
        
    except Exception as e:
        print(f"❌ Erreur de connexion timcash: {e}")
        return False

if __name__ == "__main__":
    print("🚀 Configuration MySQL pour TIM CASH")
    
    if setup_mysql_user():
        if test_timcash_connection():
            print("\n🎉 MySQL est prêt pour TIM CASH !")
            print("\n📋 Prochaines étapes:")
            print("   1. Créer les tables: python create_mysql_extended.py")
            print("   2. Ou migrer depuis SQLite: python migrate_sqlite_to_mysql.py")
        else:
            print("\n⚠️ Configuration partiellement réussie")
    else:
        print("\n💥 Échec de la configuration MySQL")
        sys.exit(1)
