"""
Schémas Pydantic pour les transferts Mobile Money
"""
from pydantic import BaseModel, Field, validator
from typing import Optional
from enum import Enum

class MobileMoneyOperator(str, Enum):
    """Opérateurs Mobile Money supportés"""
    WAVE = "wave"
    ORANGE_MONEY = "orange_money"
    MTN_MONEY = "mtn_money"
    MOOV_MONEY = "moov_money"

class MobileMoneyTransferRequest(BaseModel):
    """Requête de transfert vers Mobile Money"""
    amount: float = Field(..., gt=0, description="Montant à transférer en FCFA")
    phone: str = Field(..., min_length=8, max_length=15, description="Numéro de téléphone du destinataire")
    operator: MobileMoneyOperator = Field(..., description="Opérateur Mobile Money")
    description: Optional[str] = Field(None, max_length=200, description="Description du transfert")
    
    @validator('phone')
    def validate_phone(cls, v):
        """Valider le format du numéro de téléphone"""
        # Retirer les espaces et caractères spéciaux
        cleaned = ''.join(filter(str.isdigit, v))
        if len(cleaned) < 8:
            raise ValueError("Le numéro de téléphone doit contenir au moins 8 chiffres")
        return cleaned
    
    @validator('amount')
    def validate_amount(cls, v):
        """Valider le montant"""
        if v < 100:
            raise ValueError("Le montant minimum est de 100 FCFA")
        if v > 5000000:
            raise ValueError("Le montant maximum est de 5,000,000 FCFA")
        return v

class MobileMoneyTransferResponse(BaseModel):
    """Réponse d'un transfert Mobile Money"""
    success: bool
    transaction_id: str
    amount: float
    phone: str
    operator: str
    status: str
    message: Optional[str] = None
    cinetpay_reference: Optional[str] = None

class TransferStatusRequest(BaseModel):
    """Requête de vérification de statut"""
    transaction_id: str = Field(..., description="ID de la transaction à vérifier")

class TransferStatusResponse(BaseModel):
    """Réponse de vérification de statut"""
    success: bool
    transaction_id: str
    status: str
    amount: Optional[float] = None
    phone: Optional[str] = None
    operator: Optional[str] = None
    completed_at: Optional[str] = None
    error: Optional[str] = None

class AddContactRequest(BaseModel):
    """Requête d'ajout de contact"""
    phone: str = Field(..., description="Numéro de téléphone")
    name: str = Field(..., min_length=1, max_length=50, description="Prénom")
    surname: str = Field(..., min_length=1, max_length=50, description="Nom de famille")
    email: Optional[str] = Field(None, description="Email du contact")
    operator: MobileMoneyOperator = Field(..., description="Opérateur Mobile Money")
    
    @validator('phone')
    def validate_phone(cls, v):
        """Valider le format du numéro de téléphone"""
        cleaned = ''.join(filter(str.isdigit, v))
        if len(cleaned) < 8:
            raise ValueError("Le numéro de téléphone doit contenir au moins 8 chiffres")
        return cleaned
