"""
Modèle pour les cartes virtuelles avec couleur personnalisée
"""
from sqlalchemy import Column, String, Float, Boolean, DateTime, ForeignKey, Text, Enum as SQLEnum, Integer
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func
from database import Base
from datetime import datetime
import enum


class VirtualCardColor(str, enum.Enum):
    """Couleurs disponibles pour les cartes virtuelles - 4 couleurs à 650 FCFA"""
    GOLD = "gold"          # Or classique
    AMBER = "amber"        # Ambre (vert-jaune dégradé)
    GREEN = "green"        # Vert pur
    INFINITE = "infinite"  # Noir premium

    # Couleurs désactivées (pour compatibilité future)
    SILVER = "silver"
    BLACK = "black"
    BLUE = "blue"
    RED = "red"
    PURPLE = "purple"
    ORANGE = "orange"


class VirtualCardPurchaseStatus(str, enum.Enum):
    """Statut de l'achat de carte virtuelle"""
    PENDING = "pending"  # En attente de paiement
    PAYMENT_INITIATED = "payment_initiated"  # Paiement initié
    PAYMENT_COMPLETED = "payment_completed"  # Paiement complété
    CARD_CREATED = "card_created"  # Carte créée
    FAILED = "failed"  # Échec
    CANCELLED = "cancelled"  # Annulé


class VirtualCardPurchase(Base):
    """
    Historique des achats de cartes virtuelles
    Enregistre toutes les transactions d'achat de cartes virtuelles
    """
    __tablename__ = "virtual_card_purchases"
    
    id = Column(String, primary_key=True, index=True)
    user_id = Column(String, ForeignKey("users.id"), nullable=False)
    
    # Informations de la carte
    card_color = Column(
        SQLEnum(VirtualCardColor, values_callable=lambda x: [e.value for e in x]),
        nullable=False
    )
    card_id = Column(String, ForeignKey("nfc_cards.id"), nullable=True)  # Lien vers la carte créée
    
    # Informations de paiement
    amount = Column(Float, nullable=False)  # Prix de la carte virtuelle
    commission_amount = Column(Float, default=0.0)
    total_amount = Column(Float, nullable=False)  # Montant total payé
    
    # Statut
    status = Column(
        SQLEnum(VirtualCardPurchaseStatus, values_callable=lambda x: [e.value for e in x]),
        default=VirtualCardPurchaseStatus.PENDING
    )
    
    # Références de paiement CinetPay
    payment_transaction_id = Column(String, unique=True, index=True)  # ID transaction TIM CASH
    cinetpay_transaction_id = Column(String, nullable=True)  # ID transaction CinetPay
    cinetpay_payment_url = Column(Text, nullable=True)  # URL de paiement CinetPay
    cinetpay_payment_token = Column(String, nullable=True)
    
    # Informations supplémentaires
    payment_method = Column(String, nullable=True)  # orange_money, mtn_money, wave, etc.
    payment_phone = Column(String, nullable=True)  # Numéro utilisé pour le paiement
    
    # Dates
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    payment_initiated_at = Column(DateTime(timezone=True), nullable=True)
    payment_completed_at = Column(DateTime(timezone=True), nullable=True)
    card_created_at = Column(DateTime(timezone=True), nullable=True)
    failed_at = Column(DateTime(timezone=True), nullable=True)
    
    # Raison d'échec
    failure_reason = Column(Text, nullable=True)
    
    # Métadonnées
    device_info = Column(Text, nullable=True)  # JSON avec info appareil
    ip_address = Column(String, nullable=True)
    
    # Relations
    user = relationship("User", backref="virtual_card_purchases")
    card = relationship("NFCCard", backref="purchase_history")
    
    def __repr__(self):
        return f"<VirtualCardPurchase(id={self.id}, user_id={self.user_id}, color={self.card_color}, status={self.status})>"


class VirtualCardPrice(Base):
    """
    Configuration des prix des cartes virtuelles
    Permet de gérer les prix par couleur et par type de compte
    """
    __tablename__ = "virtual_card_prices"
    
    id = Column(String, primary_key=True, index=True)
    
    # Couleur de la carte
    card_color = Column(
        SQLEnum(VirtualCardColor, values_callable=lambda x: [e.value for e in x]),
        nullable=False,
        unique=True
    )
    
    # Prix
    base_price = Column(Float, nullable=False, default=1000.0)  # Prix de base en FCFA
    
    # Prix par type de compte (optionnel, si différent du prix de base)
    tim_mini_price = Column(Float, nullable=True)
    tim_maxi_price = Column(Float, nullable=True)
    tim_business_price = Column(Float, nullable=True)
    
    # Commission
    commission_rate = Column(Float, default=0.0)  # % de commission
    
    # Disponibilité
    is_available = Column(Boolean, default=True)
    is_premium = Column(Boolean, default=False)  # Carte premium (or, noir)
    
    # Limites
    max_purchases_per_user = Column(Integer, nullable=True)  # Limite d'achat par utilisateur
    
    # Dates
    created_at = Column(DateTime(timezone=True), server_default=func.now())
    updated_at = Column(DateTime(timezone=True), onupdate=func.now())
    
    def __repr__(self):
        return f"<VirtualCardPrice(color={self.card_color}, price={self.base_price})>"
    
    def get_price_for_account_type(self, account_type: str) -> float:
        """Retourne le prix selon le type de compte"""
        if account_type == "TIM_MINI" and self.tim_mini_price:
            return self.tim_mini_price
        elif account_type == "TIM_MAXI" and self.tim_maxi_price:
            return self.tim_maxi_price
        elif account_type == "TIM_BUSINESS" and self.tim_business_price:
            return self.tim_business_price
        return self.base_price

