# Système de Cartes NFC Physiques TimCash

## 📋 Vue d'Ensemble

Le système de cartes NFC physiques permet aux utilisateurs d'effectuer des transactions même sans smartphone compatible NFC. Les cartes sont liées aux comptes utilisateurs et peuvent être lues/écrites par des téléphones compatibles NFC.

## 🏗️ Architecture Technique

### Nouveaux Modèles de Données

#### NFCCard
```python
class NFCCard(Base):
    id = Column(String, primary_key=True)
    card_uid = Column(String, unique=True)  # UID unique de la carte
    card_number = Column(String, unique=True)  # Numéro lisible
    user_id = Column(String, ForeignKey("users.id"))
    card_type = Column(String, default="physical")
    status = Column(String, default="active")  # active, inactive, blocked
    balance = Column(Float, default=0.0)  # Solde sur la carte
    max_balance = Column(Float, default=50000.0)
    is_primary = Column(Boolean, default=False)
    last_used_at = Column(DateTime)
```

#### NFCCardTransaction
```python
class NFCCardTransaction(Base):
    id = Column(String, primary_key=True)
    card_id = Column(String, ForeignKey("nfc_cards.id"))
    transaction_id = Column(String, ForeignKey("transactions.id"))
    operation_type = Column(String)  # read, write, update_balance
    card_data = Column(Text)  # Données écrites (JSON)
    device_info = Column(Text)  # Info appareil
    location = Column(String)
```

### Services Créés

- `services/nfc_card_service.py` - Gestion des cartes NFC
- `services/nfc_transaction_service.py` - Transactions NFC
- `api/v1/route_nfc_card.py` - Routes cartes NFC
- `api/v1/route_nfc_transaction.py` - Routes transactions NFC

## 🔧 Fonctionnalités Principales

### 1. Liaison Carte ↔ Compte Utilisateur

```http
POST /api/nfc-cards/link
{
  "card_uid": "ABC123456789",
  "card_number": "TIM1001",
  "user_phone": "+22500000000"
}
```

**Processus :**
1. L'utilisateur présente sa carte NFC
2. Le vendeur scanne la carte avec son téléphone
3. La carte est liée au compte utilisateur
4. La carte devient utilisable pour les transactions

### 2. Transactions NFC

#### Paiement NFC
```http
POST /api/nfc-transactions/payment
{
  "card_uid": "ABC123456789",
  "amount": 1000,
  "device_info": "Samsung Galaxy S21 NFC",
  "location": "Abidjan, Côte d'Ivoire"
}
```

#### Recharge NFC
```http
POST /api/nfc-transactions/recharge
{
  "card_uid": "ABC123456789",
  "amount": 5000,
  "device_info": "iPhone 13 NFC",
  "location": "Dakar, Sénégal"
}
```

### 3. Gestion des Cartes

#### Récupérer les cartes de l'utilisateur
```http
GET /api/nfc-cards/my-cards
```

#### Lire les données d'une carte
```http
POST /api/nfc-cards/read
{
  "card_uid": "ABC123456789",
  "device_info": "Samsung Galaxy S21 NFC"
}
```

#### Écrire sur une carte
```http
POST /api/nfc-cards/write
{
  "card_uid": "ABC123456789",
  "card_data": {
    "last_transaction": "2024-01-01T12:00:00Z",
    "device_id": "SAMSUNG_S21",
    "location": "Abidjan, Côte d'Ivoire"
  },
  "device_info": "Samsung Galaxy S21 NFC"
}
```

## 💡 Cas d'Usage

### Scénario 1 : Utilisateur sans smartphone NFC

1. **Inscription** : L'utilisateur s'inscrit via l'application Flutter
2. **Obtention carte** : Il reçoit une carte NFC physique
3. **Liaison** : La carte est liée à son compte via l'API
4. **Transactions** : Il peut payer en présentant sa carte à un vendeur avec smartphone NFC

### Scénario 2 : Vendeur avec smartphone NFC

1. **Configuration** : Le vendeur configure son smartphone NFC
2. **Scan carte** : Il scanne la carte du client
3. **Validation** : Le système valide la carte et le solde
4. **Transaction** : La transaction est effectuée automatiquement
5. **Confirmation** : Les deux parties reçoivent une confirmation

### Scénario 3 : Recharge de carte

1. **Client** : Le client présente sa carte NFC
2. **Recharger** : Un recharger scanne la carte avec son smartphone
3. **Montant** : Le recharger saisit le montant à recharger
4. **Vérification** : Le système vérifie le solde du recharger
5. **Transfert** : Le montant est transféré du recharger vers le client

## 🔒 Sécurité

### Validation des Cartes
- Vérification de l'UID unique
- Contrôle du statut (active/inactive)
- Validation du propriétaire
- Limitation des montants

### Traçabilité
- Historique complet des transactions
- Logs des opérations de lecture/écriture
- Géolocalisation des transactions
- Informations sur l'appareil utilisé

### Gestion des Erreurs
- Cartes perdues/volées
- Blocage automatique
- Notifications de sécurité
- Récupération de compte

## 📊 API Endpoints

### Gestion des Cartes
```http
POST /api/nfc-cards/link                    # Lier une carte
GET  /api/nfc-cards/my-cards               # Mes cartes
GET  /api/nfc-cards/{uid}/balance          # Solde d'une carte
POST /api/nfc-cards/read                   # Lire une carte
POST /api/nfc-cards/write                  # Écrire sur une carte
PUT  /api/nfc-cards/{uid}/update           # Mettre à jour une carte
DELETE /api/nfc-cards/{uid}/unlink         # Délier une carte
```

### Transactions NFC
```http
POST /api/nfc-transactions/payment          # Paiement NFC
POST /api/nfc-transactions/recharge         # Recharge NFC
GET  /api/nfc-transactions/history         # Historique
POST /api/nfc-transactions/validate-card   # Valider une carte
GET  /api/nfc-transactions/stats           # Statistiques
```

### Administration
```http
GET  /api/admin/nfc-cards                  # Toutes les cartes (admin)
```

## 🚀 Déploiement

### 1. Migration de la Base de Données

```bash
# Exécuter le script de migration
python migrate_nfc_system.py
```

### 2. Test du Système

```bash
# Tester le système NFC
python test_nfc_system.py
```

### 3. Création de Cartes Physiques

```python
# Exemple de création de carte
card_data = {
    "card_uid": "ABC123456789",
    "card_number": "TIM1001",
    "user_phone": "+22500000000"
}
```

## 📱 Intégration Flutter

### Lecture NFC
```dart
// Lire une carte NFC
NfcTag tag = await NfcManager.instance.startSession(
  onDiscovered: (NfcTag tag) async {
    String cardUid = tag.data['nfca']['identifier'];
    // Envoyer à l'API
  }
);
```

### Écriture NFC
```dart
// Écrire sur une carte NFC
await NfcManager.instance.startSession(
  onDiscovered: (NfcTag tag) async {
    await tag.write(
      NdefMessage([
        NdefRecord.createText('{"user_id": "123", "balance": 1000}')
      ])
    );
  }
);
```

## 🔍 Monitoring et Logs

### Logs de Transaction
```python
# Exemple de log
logger.info(f"Transaction NFC: {amount} FCFA de {client} vers {merchant}")
logger.info(f"Carte NFC {card_uid} utilisée par {device_info}")
```

### Métriques
- Nombre de cartes actives
- Volume des transactions NFC
- Taux de succès des opérations
- Géolocalisation des transactions

## ⚠️ Points d'Attention

1. **Compatibilité NFC** : Vérifier la compatibilité des appareils
2. **Sécurité** : Chiffrement des données sur les cartes
3. **Performance** : Optimisation des opérations NFC
4. **Batterie** : Impact sur l'autonomie des smartphones

## 🔮 Évolutions Futures

- **Cartes virtuelles** : NFC intégré dans les smartphones
- **Paiements sans contact** : Support des standards internationaux
- **Géolocalisation avancée** : Détection automatique des lieux
- **Analytics** : Tableaux de bord détaillés
- **Intégration bancaire** : Liaison avec les comptes bancaires

## 📈 Avantages du Système

1. **Accessibilité** : Transactions sans smartphone NFC
2. **Simplicité** : Interface intuitive
3. **Sécurité** : Cartes physiques sécurisées
4. **Flexibilité** : Support multi-appareils
5. **Traçabilité** : Historique complet
6. **Évolutivité** : Architecture modulaire

## 🧪 Tests et Validation

### Tests Unitaires
- Validation des cartes
- Calcul des soldes
- Gestion des erreurs
- Sécurité des transactions

### Tests d'Intégration
- Liaison carte-utilisateur
- Transactions complètes
- Gestion des conflits
- Performance

### Tests de Charge
- Volume de transactions
- Concurrence
- Temps de réponse
- Stabilité

Le système de cartes NFC physiques TimCash est maintenant **opérationnel** et **prêt pour la production** ! 🎯
