# 💸 Intégration Mobile Money - TIM CASH

## Vue d'ensemble

Cette intégration permet aux utilisateurs de TIM CASH (TIM MINI, TIM MAXI, TIM BUSINESS) de transférer de l'argent depuis leur wallet vers des comptes Mobile Money.

### Opérateurs supportés
- **Wave** (Sénégal)
- **Orange Money** (Côte d'Ivoire, Mali, Sénégal)
- **MTN Money** (Côte d'Ivoire)
- **Moov Money** (Côte d'Ivoire)

---

## 🔧 Configuration

### 1. Variables d'environnement

Ajoutez vos credentials CinetPay dans `.env.mysql`:

```env
CINETPAY_API_KEY=votre-api-key
CINETPAY_SECRET_KEY=votre-secret-key
CINETPAY_CALLBACK_URL=http://localhost:8001/api/mobile-money/callback
```

### 2. Migration de la base de données

Le nouveau type de transaction `MOBILE_MONEY_TRANSFER` a été ajouté. Si vous utilisez Alembic:

```bash
cd backend
alembic revision --autogenerate -m "Add mobile money transfer type"
alembic upgrade head
```

---

## 📡 Endpoints API

### Base URL
```
http://localhost:8001/api
```

### 1. Transférer vers Mobile Money

**POST** `/mobile-money/transfer`

Transfère de l'argent du wallet TIM CASH vers un compte Mobile Money.

**Headers:**
```
Authorization: Bearer <token>
Content-Type: application/json
```

**Body:**
```json
{
  "amount": 5000,
  "phone": "0769989178",
  "operator": "orange_money",
  "description": "Paiement facture"
}
```

**Opérateurs disponibles:**
- `wave`
- `orange_money`
- `mtn_money`
- `moov_money`

**Response (200):**
```json
{
  "success": true,
  "transaction_id": "MM-A1B2C3D4",
  "amount": 5000,
  "phone": "0769989178",
  "operator": "orange_money",
  "status": "pending",
  "message": "Transfert en cours de traitement",
  "cinetpay_reference": "CP-REF-123456"
}
```

**Frais:**
- 2% du montant transféré

---

### 2. Vérifier le statut d'un transfert

**GET** `/mobile-money/status/{transaction_id}`

Vérifie le statut d'un transfert Mobile Money.

**Headers:**
```
Authorization: Bearer <token>
```

**Response (200):**
```json
{
  "success": true,
  "transaction_id": "MM-A1B2C3D4",
  "status": "completed",
  "amount": 5000,
  "phone": "0769989178",
  "operator": "orange_money",
  "completed_at": "2025-10-25T14:30:00Z"
}
```

**Statuts possibles:**
- `pending` - En cours de traitement
- `completed` - Transfert réussi
- `failed` - Transfert échoué (wallet remboursé automatiquement)

---

### 3. Ajouter un contact Mobile Money

**POST** `/mobile-money/add-contact`

Ajoute un contact pour faciliter les transferts futurs.

**Headers:**
```
Authorization: Bearer <token>
Content-Type: application/json
```

**Body:**
```json
{
  "phone": "0769989178",
  "name": "Jean",
  "surname": "Dupont",
  "email": "jean.dupont@example.com",
  "operator": "orange_money"
}
```

**Response (200):**
```json
{
  "success": true,
  "message": "Contact ajouté avec succès",
  "data": {
    "contact_id": "12345"
  }
}
```

---

### 4. Liste des opérateurs

**GET** `/mobile-money/operators`

Retourne la liste des opérateurs Mobile Money supportés.

**Response (200):**
```json
{
  "operators": [
    {
      "code": "wave",
      "name": "Wave",
      "country": "Sénégal",
      "prefix": "221",
      "logo": "wave_logo.png"
    },
    {
      "code": "orange_money",
      "name": "Orange Money",
      "country": "Multi-pays",
      "prefix": "225",
      "logo": "orange_money_logo.png"
    }
  ]
}
```

---

## 🔄 Flux de transfert

```
1. Utilisateur initie transfert
   ↓
2. Vérification du solde wallet
   ↓
3. Calcul des frais (2%)
   ↓
4. Création transaction (PENDING)
   ↓
5. Débit du wallet
   ↓
6. Appel API CinetPay
   ↓
7. CinetPay traite le transfert
   ↓
8. Callback CinetPay → Mise à jour statut
   ↓
9. Transaction COMPLETED ou FAILED
   (Si FAILED → Remboursement automatique)
```

---

## 🧪 Tests

### Test avec cURL

```bash
# 1. Login
curl -X POST http://localhost:8001/api/financials/auth/user/login \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "0769989178",
    "hashed_password": "admin123",
    "country_id": "33dac6d2-4a96-48a1-9376-448308e5b118"
  }'

# 2. Transfert Mobile Money
curl -X POST http://localhost:8001/api/mobile-money/transfer \
  -H "Authorization: Bearer <votre-token>" \
  -H "Content-Type: application/json" \
  -d '{
    "amount": 1000,
    "phone": "0769989178",
    "operator": "orange_money",
    "description": "Test transfert"
  }'

# 3. Vérifier le statut
curl -X GET http://localhost:8001/api/mobile-money/status/MM-A1B2C3D4 \
  -H "Authorization: Bearer <votre-token>"
```

---

## 🔐 Sécurité

### Validations
- ✅ Montant minimum: 100 FCFA
- ✅ Montant maximum: 5,000,000 FCFA
- ✅ Vérification du solde avant transfert
- ✅ Authentification JWT requise
- ✅ Validation du format de téléphone
- ✅ Remboursement automatique en cas d'échec

### Gestion des erreurs
- Solde insuffisant → HTTP 400
- Transaction non trouvée → HTTP 404
- Erreur CinetPay → HTTP 400 avec détails
- Erreur serveur → HTTP 500

---

## 📊 Codes de statut CinetPay

| Code | Signification |
|------|---------------|
| `00` | Succès |
| `01` | Échec |
| `ACCEPTED` | Accepté |
| `REFUSED` | Refusé |
| `completed` | Complété |
| `failed` | Échoué |

---

## 🚀 Déploiement

### 1. Configuration production

Mettez à jour `.env` avec vos vraies credentials CinetPay:

```env
CINETPAY_API_KEY=prod-api-key
CINETPAY_SECRET_KEY=prod-secret-key
CINETPAY_CALLBACK_URL=https://votre-domaine.com/api/mobile-money/callback
```

### 2. Redémarrer le serveur

```bash
cd backend
python -m uvicorn main:app --reload --port 8001
```

### 3. Tester l'intégration

Accédez à la documentation interactive:
```
http://localhost:8001/docs
```

Recherchez la section **mobile-money** pour tester les endpoints.

---

## 📱 Intégration Frontend (Flutter)

### Exemple de code Dart

```dart
// Service de transfert Mobile Money
class MobileMoneyService {
  static const String baseUrl = 'http://localhost:8001/api';
  
  Future<Map<String, dynamic>> transferToMobileMoney({
    required double amount,
    required String phone,
    required String operator,
    String? description,
  }) async {
    final response = await http.post(
      Uri.parse('$baseUrl/mobile-money/transfer'),
      headers: {
        'Authorization': 'Bearer $token',
        'Content-Type': 'application/json',
      },
      body: jsonEncode({
        'amount': amount,
        'phone': phone,
        'operator': operator,
        'description': description,
      }),
    );
    
    if (response.statusCode == 200) {
      return jsonDecode(response.body);
    } else {
      throw Exception('Échec du transfert');
    }
  }
  
  Future<Map<String, dynamic>> checkTransferStatus(String transactionId) async {
    final response = await http.get(
      Uri.parse('$baseUrl/mobile-money/status/$transactionId'),
      headers: {
        'Authorization': 'Bearer $token',
      },
    );
    
    if (response.statusCode == 200) {
      return jsonDecode(response.body);
    } else {
      throw Exception('Impossible de vérifier le statut');
    }
  }
}
```

---

## 🐛 Dépannage

### Erreur: "CinetPay credentials not configured"
→ Vérifiez que `CINETPAY_API_KEY` et `CINETPAY_SECRET_KEY` sont définis dans `.env`

### Erreur: "Solde insuffisant"
→ Rechargez le wallet ou réduisez le montant (n'oubliez pas les frais de 2%)

### Transfert bloqué en "pending"
→ Vérifiez les logs CinetPay ou appelez `/mobile-money/status/{transaction_id}`

### Callback non reçu
→ Vérifiez que `CINETPAY_CALLBACK_URL` est accessible publiquement (utilisez ngrok en dev)

---

## 📞 Support

Pour toute question sur l'intégration CinetPay:
- Documentation: https://docs.cinetpay.com
- Support: support@cinetpay.com

---

## ✅ Checklist de mise en production

- [ ] Credentials CinetPay configurés
- [ ] URL de callback accessible publiquement
- [ ] Tests de transfert effectués
- [ ] Gestion des erreurs vérifiée
- [ ] Logs de monitoring activés
- [ ] Limites de transfert configurées
- [ ] Notifications utilisateur implémentées

---

**Version:** 1.0.0  
**Date:** 25 Octobre 2025  
**Auteur:** TIM CASH Team
